%---------------------------------------------------------------------------%
% vim: ft=mercury ts=4 sw=4 et
%---------------------------------------------------------------------------%
% File: render.m
%
% SVG rendering of mazes.
%
% 🤖 Generated with [Claude Code](https://claude.ai/code)
%
% Co-authored-by: Claude <noreply@anthropic.com>
%
%---------------------------------------------------------------------------%

:- module render.
:- interface.

:- import_module grid.
:- import_module io.
:- import_module list.
:- import_module maze.

%---------------------------------------------------------------------------%

    % Rendering options.
    %
:- type render_options
    --->    render_options(
                ro_cell_spacing :: int,
                ro_margin :: int,
                ro_maze_bg_color :: string,
                ro_cell_bg_color :: string,
                ro_internal_wall_color :: string,
                ro_boundary_wall_color :: string,
                ro_path_color :: string,
                ro_internal_wall_width :: int,
                ro_boundary_wall_width :: int,
                ro_path_width :: int
            ).

    % Default rendering options.
    %
:- func default_render_options = render_options.

    % render_maze(Filename, Options, Maze, Solution, !IO)
    %
    % Render the maze to an SVG file.
    % Solution is the list of cells in the solution path (may be empty).
    %
:- pred render_maze(string::in, render_options::in, maze::in, list(cell)::in,
    io::di, io::uo) is det.

%---------------------------------------------------------------------------%
%---------------------------------------------------------------------------%

:- implementation.

:- import_module boundary.
:- import_module float.
:- import_module int.
:- import_module math.
:- import_module options.
:- import_module require.
:- import_module string.

%---------------------------------------------------------------------------%

default_render_options = render_options(
    10,         % cell_spacing
    2,          % margin
    "none",     % maze_bg_color
    "white",    % cell_bg_color
    "black",    % internal_wall_color
    "black",    % boundary_wall_color
    "red",      % path_color
    1,          % internal_wall_width
    2,          % boundary_wall_width
    3           % path_width
).

%---------------------------------------------------------------------------%

    % Constant for hex geometry: sqrt(3).
    %
:- func sqrt3 = float.
sqrt3 = math.sqrt(3.0).

%---------------------------------------------------------------------------%
%
% Hex corner geometry.
%
% A flat-topped hexagon has 6 corners, named by their position relative
% to the centre. Going counter-clockwise from the right:
%
%              upper_left ----- upper_right
%                       /         \
%                      /           \
%                 left               right
%                      \           /
%                       \         /
%              lower_left ----- lower_right
%

:- type hex_corner
    --->    right
    ;       upper_right
    ;       upper_left
    ;       left
    ;       lower_left
    ;       lower_right.

    % Return the offset of a hex corner from the cell centre.
    % Sf is the cell spacing (distance from centre to left/right corners).
    %
:- pred hex_corner_offset(float::in, hex_corner::in,
    float::out, float::out) is det.

hex_corner_offset(Sf, Corner, DX, DY) :-
    (
        Corner = right,
        DX = Sf,
        DY = 0.0
    ;
        Corner = upper_right,
        DX = Sf / 2.0,
        DY = -Sf * sqrt3 / 2.0
    ;
        Corner = upper_left,
        DX = -Sf / 2.0,
        DY = -Sf * sqrt3 / 2.0
    ;
        Corner = left,
        DX = -Sf,
        DY = 0.0
    ;
        Corner = lower_left,
        DX = -Sf / 2.0,
        DY = Sf * sqrt3 / 2.0
    ;
        Corner = lower_right,
        DX = Sf / 2.0,
        DY = Sf * sqrt3 / 2.0
    ).

    % Return the two corners that bound an edge in the given direction.
    % The corners are ordered for counter-clockwise traversal around the cell
    % (which corresponds to clockwise traversal around the maze boundary).
    %
:- pred hex_edge_corners(direction::in, hex_corner::out, hex_corner::out)
    is det.

hex_edge_corners(Dir, StartCorner, EndCorner) :-
    (
        Dir = up,
        StartCorner = upper_left,
        EndCorner = upper_right
    ;
        Dir = upper_right,
        StartCorner = upper_right,
        EndCorner = right
    ;
        Dir = lower_right,
        StartCorner = right,
        EndCorner = lower_right
    ;
        Dir = down,
        StartCorner = lower_right,
        EndCorner = lower_left
    ;
        Dir = lower_left,
        StartCorner = lower_left,
        EndCorner = left
    ;
        Dir = upper_left,
        StartCorner = left,
        EndCorner = upper_left
    ;
        % Square directions - not valid for hex, but we need complete coverage
        ( Dir = left ; Dir = right ),
        StartCorner = right,
        EndCorner = right
    ).

    % Calculate viewbox dimensions based on topology.
    %
:- pred viewbox_dimensions(topology::in, bounds::in, int::in, int::in,
    int::out, int::out) is det.

viewbox_dimensions(square, Bounds, S, M, ViewWidth, ViewHeight) :-
    W = Bounds ^ width,
    H = Bounds ^ height,
    ViewWidth = (2 * M + W) * S,
    ViewHeight = (2 * M + H) * S.

viewbox_dimensions(hex, Bounds, S, M, ViewWidth, ViewHeight) :-
    W = Bounds ^ width,
    H = Bounds ^ height,
    Sf = float.float(S),
    Mf = float.float(M),
    % Hex bounding box calculation:
    % The maze spans from Q=0 to Q=W-1 horizontally.
    % Cell centres are at X = Q * 1.5 * S + offset
    % Each cell extends S in each direction from centre.
    % Width = (W-1) * 1.5 * S + 2 * S = ((W-1) * 1.5 + 2) * S
    HexContentWidth = (float.float(W - 1) * 1.5 + 2.0) * Sf,
    % The maze spans from R=0 to R=(W-1)/2+H-1 vertically.
    % But the hex shape means not all columns have the same height.
    % The total vertical span is H cells on each vertical edge,
    % plus the diagonal offset. Actually it's simpler:
    % The leftmost column (Q=0) has R from 0 to H-1, so H cells.
    % The rightmost column (Q=W-1) has R from (W-1)/2 to (W-1)/2+H-1, so H cells.
    % The vertical extent in screen coords is H * sqrt(3) * S + S * sqrt(3)/2
    % (accounting for cell radius).
    % Actually, we need to find the actual min/max Y across all cells.
    % Let's use a simpler approach: max_y - min_y + 2*S for cell radius.
    HalfW = (W - 1) / 2,
    MaxR = HalfW + H - 1,
    % Y = (R - Q/2) * sqrt3 * S + offset
    % For Q=0, R=0: Y = 0
    % For Q=HalfW, R=0: Y = -HalfW/2 * sqrt3 * S (topmost)
    % For Q=0, R=H-1: Y = (H-1) * sqrt3 * S
    % For Q=W-1, R=MaxR: Y = (MaxR - (W-1)/2) * sqrt3 * S = (H-1) * sqrt3 * S
    % Min Y is at 12 o'clock corner (Q=HalfW, R=0): -HalfW/2 * sqrt3 * S
    % Max Y is at 6 o'clock corner (Q=HalfW, R=MaxR): MaxR - HalfW/2
    MinYOffset = -float.float(HalfW) / 2.0 * sqrt3,
    MaxYOffset = (float.float(MaxR) - float.float(HalfW) / 2.0) * sqrt3,
    HexContentHeight = (MaxYOffset - MinYOffset) * Sf + 2.0 * Sf * sqrt3 / 2.0,
    ViewWidth = float.truncate_to_int(HexContentWidth + 2.0 * Mf * Sf + 0.5),
    ViewHeight = float.truncate_to_int(HexContentHeight + 2.0 * Mf * Sf + 0.5).

%---------------------------------------------------------------------------%

render_maze(Filename, RenderOpts, Maze, Solution, !IO) :-
    io.open_output(Filename, Result, !IO),
    (
        Result = ok(Stream),
        render_to_stream(Stream, RenderOpts, Maze, Solution, !IO),
        io.close_output(Stream, !IO)
    ;
        Result = error(Error),
        io.error_message(Error, Msg),
        io.stderr_stream(Stderr, !IO),
        io.format(Stderr, "Error opening %s: %s\n", [s(Filename), s(Msg)], !IO)
    ).

:- pred render_to_stream(io.text_output_stream::in, render_options::in,
    maze::in, list(cell)::in, io::di, io::uo) is det.

render_to_stream(Stream, RenderOpts, Maze, Solution, !IO) :-
    Topology = get_topology(Maze),
    Bounds = get_bounds(Maze),
    S = RenderOpts ^ ro_cell_spacing,
    M = RenderOpts ^ ro_margin,

    % Calculate viewbox dimensions based on topology
    viewbox_dimensions(Topology, Bounds, S, M, ViewWidth, ViewHeight),

    % SVG header
    io.write_string(Stream, "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n", !IO),
    io.format(Stream,
        "<svg xmlns=\"http://www.w3.org/2000/svg\" viewBox=\"0 0 %d %d\">\n",
        [i(ViewWidth), i(ViewHeight)], !IO),

    % Layer 1: Maze background
    render_maze_background(Stream, RenderOpts, ViewWidth, ViewHeight, !IO),

    % Layer 2: Cell background (boundary polygon filled)
    render_cell_background(Stream, RenderOpts, Maze, !IO),

    % Layer 3: Internal walls
    render_internal_walls(Stream, RenderOpts, Maze, !IO),

    % Layer 4: Boundary walls
    render_boundary_walls(Stream, RenderOpts, Maze, !IO),

    % Layer 5: Solution path
    render_solution(Stream, RenderOpts, Maze, Solution, !IO),

    % SVG footer
    io.write_string(Stream, "</svg>\n", !IO).

%---------------------------------------------------------------------------%

:- pred render_maze_background(io.text_output_stream::in, render_options::in,
    int::in, int::in, io::di, io::uo) is det.

render_maze_background(Stream, RenderOpts, ViewWidth, ViewHeight, !IO) :-
    BgColor = RenderOpts ^ ro_maze_bg_color,
    ( if BgColor = "none" then
        true
    else
        io.format(Stream,
            "  <rect x=\"0\" y=\"0\" width=\"%d\" height=\"%d\" fill=\"%s\"/>\n",
            [i(ViewWidth), i(ViewHeight), s(BgColor)], !IO)
    ).

%---------------------------------------------------------------------------%

:- pred render_cell_background(io.text_output_stream::in, render_options::in,
    maze::in, io::di, io::uo) is det.

render_cell_background(Stream, RenderOpts, Maze, !IO) :-
    Topology = get_topology(Maze),
    Bounds = get_bounds(Maze),
    S = RenderOpts ^ ro_cell_spacing,
    M = RenderOpts ^ ro_margin,
    (
        Topology = square,
        % Draw a rectangle covering all cells
        W = Bounds ^ width,
        H = Bounds ^ height,
        X = M * S,
        Y = M * S,
        RectW = W * S,
        RectH = H * S,
        io.format(Stream,
            "  <rect x=\"%d\" y=\"%d\" width=\"%d\" height=\"%d\" fill=\"%s\"/>\n",
            [i(X), i(Y), i(RectW), i(RectH), s(RenderOpts ^ ro_cell_bg_color)],
            !IO)
    ;
        Topology = hex,
        % Draw a polygon covering all cells (the hex maze boundary)
        render_hex_background_polygon(Stream, RenderOpts, Bounds, !IO)
    ).

    % Render the background polygon for a hex maze.
    % The polygon traces the outer boundary of the pointy-topped hexagon shape.
    %
:- pred render_hex_background_polygon(io.text_output_stream::in,
    render_options::in, bounds::in, io::di, io::uo) is det.

render_hex_background_polygon(Stream, RenderOpts, Bounds, !IO) :-
    W = Bounds ^ width,
    H = Bounds ^ height,
    S = RenderOpts ^ ro_cell_spacing,
    M = RenderOpts ^ ro_margin,
    Sf = float.float(S),
    HalfW = (W - 1) / 2,

    % The six corner cells and their outermost hex corners:
    % 10 o'clock (0, 0): left corner
    % 12 o'clock (HalfW, 0): top point (between upper-left and upper-right)
    % 2 o'clock (W-1, HalfW): right corner
    % 4 o'clock (W-1, HalfW + H - 1): right corner
    % 6 o'clock (HalfW, HalfW + H - 1): bottom point (between lower-left and lower-right)
    % 8 o'clock (0, H - 1): left corner

    % Get the cell centres for each corner of the maze
    hex_cell_centre(S, M, Bounds, cell(0, 0), C10X, C10Y),
    hex_cell_centre(S, M, Bounds, cell(HalfW, 0), C12X, C12Y),
    hex_cell_centre(S, M, Bounds, cell(W - 1, HalfW), C2X, C2Y),
    hex_cell_centre(S, M, Bounds, cell(W - 1, HalfW + H - 1), C4X, C4Y),
    hex_cell_centre(S, M, Bounds, cell(HalfW, HalfW + H - 1), C6X, C6Y),
    hex_cell_centre(S, M, Bounds, cell(0, H - 1), C8X, C8Y),

    % Calculate the 6 polygon vertices using corner offsets where applicable
    % 10 o'clock: left corner of cell (0, 0)
    hex_corner_offset(Sf, left, D10X, D10Y),
    P10X = C10X + D10X,
    P10Y = C10Y + D10Y,
    % 12 o'clock: top point (at upper corner Y, cell centre X)
    hex_corner_offset(Sf, upper_right, _, D12Y),
    P12X = C12X,
    P12Y = C12Y + D12Y,
    % 2 o'clock: right corner of cell (W-1, HalfW)
    hex_corner_offset(Sf, right, D2X, D2Y),
    P2X = C2X + D2X,
    P2Y = C2Y + D2Y,
    % 4 o'clock: right corner of cell (W-1, HalfW + H - 1)
    hex_corner_offset(Sf, right, D4X, D4Y),
    P4X = C4X + D4X,
    P4Y = C4Y + D4Y,
    % 6 o'clock: bottom point (at lower corner Y, cell centre X)
    hex_corner_offset(Sf, lower_right, _, D6Y),
    P6X = C6X,
    P6Y = C6Y + D6Y,
    % 8 o'clock: left corner of cell (0, H - 1)
    hex_corner_offset(Sf, left, D8X, D8Y),
    P8X = C8X + D8X,
    P8Y = C8Y + D8Y,

    % Format the polygon points
    PointsStr = string.format("%s,%s %s,%s %s,%s %s,%s %s,%s %s,%s",
        [s(format_coord(P10X)), s(format_coord(P10Y)),
         s(format_coord(P12X)), s(format_coord(P12Y)),
         s(format_coord(P2X)), s(format_coord(P2Y)),
         s(format_coord(P4X)), s(format_coord(P4Y)),
         s(format_coord(P6X)), s(format_coord(P6Y)),
         s(format_coord(P8X)), s(format_coord(P8Y))]),

    io.format(Stream,
        "  <polygon points=\"%s\" fill=\"%s\"/>\n",
        [s(PointsStr), s(RenderOpts ^ ro_cell_bg_color)], !IO).

    % Calculate the screen coordinates of a hex cell centre.
    %
:- pred hex_cell_centre(int::in, int::in, bounds::in, cell::in,
    float::out, float::out) is det.

hex_cell_centre(S, M, Bounds, cell(Q, R), CX, CY) :-
    W = Bounds ^ width,
    Sf = float.float(S),
    Mf = float.float(M),
    Qf = float.float(Q),
    Rf = float.float(R),
    HalfW = float.float((W - 1) / 2),

    % X position: Q * 1.5 * S, offset so leftmost cell is at margin + S
    CX = Mf * Sf + Sf + Qf * 1.5 * Sf,

    % Y position: need to account for the skew
    % Cell (0, 0) is at 10 o'clock, which is not the top of the viewbox.
    % The topmost point is at 12 o'clock (cell (HalfW, 0)).
    % Y = (R - Q/2) * sqrt3 * S, but we need to offset so the top is at margin.
    % The min Y offset (before margin) is at Q=HalfW, R=0: -HalfW/2 * sqrt3 * S
    % So we add that amount to bring the top to Y=0, then add margin.
    MinYOffset = -HalfW / 2.0 * sqrt3 * Sf,
    CY = Mf * Sf + Sf * sqrt3 / 2.0 + (Rf - Qf / 2.0) * sqrt3 * Sf - MinYOffset.

%---------------------------------------------------------------------------%

:- pred render_internal_walls(io.text_output_stream::in, render_options::in,
    maze::in, io::di, io::uo) is det.

render_internal_walls(Stream, RenderOpts, Maze, !IO) :-
    Topology = get_topology(Maze),
    Bounds = get_bounds(Maze),
    AllEdges = internal_edges(Maze),
    % Walls are edges without doors
    Walls = list.filter(pred(E::in) is semidet :- not has_door(Maze, E),
        AllEdges),
    list.foldl(render_wall(Stream, RenderOpts, Topology, Bounds), Walls, !IO).

:- pred render_wall(io.text_output_stream::in, render_options::in,
    topology::in, bounds::in, edge::in, io::di, io::uo) is det.

render_wall(Stream, RenderOpts, Topology, Bounds, Edge, !IO) :-
    S = RenderOpts ^ ro_cell_spacing,
    M = RenderOpts ^ ro_margin,
    edge_cells(Topology, Edge, Cell1, Cell2),
    edge_endpoints(Topology, Bounds, S, M, Cell1, Cell2, X1, Y1, X2, Y2),
    io.format(Stream,
        "  <line x1=\"%s\" y1=\"%s\" x2=\"%s\" y2=\"%s\" stroke=\"%s\" stroke-width=\"%d\" stroke-linecap=\"round\"/>\n",
        [s(format_coord(X1)), s(format_coord(Y1)),
         s(format_coord(X2)), s(format_coord(Y2)),
         s(RenderOpts ^ ro_internal_wall_color),
         i(RenderOpts ^ ro_internal_wall_width)], !IO).

    % Calculate the endpoints of an edge between two adjacent cells.
    %
:- pred edge_endpoints(topology::in, bounds::in, int::in, int::in,
    cell::in, cell::in, float::out, float::out, float::out, float::out) is det.

edge_endpoints(square, _Bounds, S, M, cell(X1, Y1), cell(X2, Y2),
        PX1, PY1, PX2, PY2) :-
    Sf = float.float(S),
    Mf = float.float(M),
    ( if X1 = X2 then
        % Horizontal edge (cells are vertically adjacent)
        X = float.float(X1),
        Y = float.float(int.max(Y1, Y2)),
        PX1 = (Mf + X) * Sf,
        PY1 = (Mf + Y) * Sf,
        PX2 = (Mf + X + 1.0) * Sf,
        PY2 = (Mf + Y) * Sf
    else
        % Vertical edge (cells are horizontally adjacent)
        X = float.float(int.max(X1, X2)),
        Y = float.float(Y1),
        PX1 = (Mf + X) * Sf,
        PY1 = (Mf + Y) * Sf,
        PX2 = (Mf + X) * Sf,
        PY2 = (Mf + Y + 1.0) * Sf
    ).

edge_endpoints(hex, Bounds, S, M, Cell1, Cell2, PX1, PY1, PX2, PY2) :-
    % For hex grids, find the direction from Cell1 to Cell2 and calculate
    % the shared edge endpoints based on hex geometry.
    Cell1 = cell(Q1, R1),
    Cell2 = cell(Q2, R2),
    DQ = Q2 - Q1,
    DR = R2 - R1,
    hex_cell_centre(S, M, Bounds, Cell1, CX, CY),
    Sf = float.float(S),
    % Find the direction that matches this delta
    Dir = delta_to_direction(hex, DQ, DR),
    % Calculate edge endpoints based on direction
    hex_edge_endpoints(Sf, CX, CY, Dir, PX1, PY1, PX2, PY2).

    % Convert a coordinate delta to the corresponding direction.
    %
:- func delta_to_direction(topology, int, int) = direction.

delta_to_direction(Topology, DX, DY) = Dir :-
    Dirs = directions(Topology),
    ( if find_matching_direction(Topology, Dirs, DX, DY, D) then
        Dir = D
    else
        unexpected($pred, "no direction matches delta")
    ).

:- pred find_matching_direction(topology::in, list(direction)::in,
    int::in, int::in, direction::out) is semidet.

find_matching_direction(Topology, [D | Ds], DX, DY, Dir) :-
    ( if direction_delta(Topology, D, DX, DY) then
        Dir = D
    else
        find_matching_direction(Topology, Ds, DX, DY, Dir)
    ).

    % Calculate hex edge endpoints for a given direction.
    % The direction is from the cell centre outward.
    % Uses hex_edge_corners to determine which corners bound the edge.
    %
:- pred hex_edge_endpoints(float::in, float::in, float::in, direction::in,
    float::out, float::out, float::out, float::out) is det.

hex_edge_endpoints(Sf, CX, CY, Dir, PX1, PY1, PX2, PY2) :-
    hex_edge_corners(Dir, Corner1, Corner2),
    hex_corner_offset(Sf, Corner1, DX1, DY1),
    hex_corner_offset(Sf, Corner2, DX2, DY2),
    PX1 = CX + DX1,
    PY1 = CY + DY1,
    PX2 = CX + DX2,
    PY2 = CY + DY2.

:- func format_coord(float) = string.

format_coord(F) = Str :-
    ( if floor(F) = F then
        Str = string.from_int(float.truncate_to_int(F))
    else
        Str = string.format("%.1f", [f(F)])
    ).

%---------------------------------------------------------------------------%

:- pred render_boundary_walls(io.text_output_stream::in, render_options::in,
    maze::in, io::di, io::uo) is det.

render_boundary_walls(Stream, RenderOpts, Maze, !IO) :-
    Topology = get_topology(Maze),
    Bounds = get_bounds(Maze),
    Entrance = get_entrance(Maze),
    Exit = get_exit(Maze),

    % Get all boundary walls and split at entrance/exit
    AllWalls = boundary_walls(Topology, Bounds),
    split_boundary(AllWalls, Entrance, Exit, Segment1, Segment2),

    % Segment1: from entrance to exit (clockwise)
    % Segment2: from exit to entrance (clockwise)
    % Each segment excludes the entrance/exit openings

    % Render first polyline: starts at entrance end point, goes through Segment1,
    % ends at exit start point
    render_boundary_polyline(Stream, RenderOpts, Topology, Bounds,
        Entrance, Segment1, Exit, !IO),

    % Render second polyline: starts at exit end point, goes through Segment2,
    % ends at entrance start point
    render_boundary_polyline(Stream, RenderOpts, Topology, Bounds,
        Exit, Segment2, Entrance, !IO).

    % Render a boundary polyline from the end of StartWall, through the segment,
    % to the start of EndWall.
    %
:- pred render_boundary_polyline(io.text_output_stream::in, render_options::in,
    topology::in, bounds::in,
    boundary_wall::in, list(boundary_wall)::in, boundary_wall::in,
    io::di, io::uo) is det.

render_boundary_polyline(Stream, RenderOpts, Topology, Bounds,
        StartWall, Segment, _EndWall, !IO) :-
    S = RenderOpts ^ ro_cell_spacing,
    M = RenderOpts ^ ro_margin,
    % The polyline traces from the end of the StartWall (where its gap ends),
    % through all edges in Segment, to the start of EndWall (where its gap begins).
    %
    % Since consecutive edges share vertices, we only need:
    % - Start point: end of StartWall
    % - For each edge in Segment: its end point (which is start of next edge)
    % - Note: start of first segment edge = end of StartWall (already included)
    % - Note: end of last segment edge = start of EndWall
    %
    % So we output: end(StartWall), then end of each segment edge.
    % The last segment edge's end = start(EndWall), so we don't need EndPoint separately.
    %
    StartPoint = wall_end_point(Topology, Bounds, S, M, StartWall),
    SegmentEndPoints = list.map(
        (func(W) = wall_end_point(Topology, Bounds, S, M, W)),
        Segment),
    AllPoints = [StartPoint | SegmentEndPoints],
    PointsStr = string.join_list(",", AllPoints),
    io.format(Stream,
        "  <polyline points=\"%s\" fill=\"none\" stroke=\"%s\" stroke-width=\"%d\" stroke-linecap=\"round\" stroke-linejoin=\"round\"/>\n",
        [s(PointsStr), s(RenderOpts ^ ro_boundary_wall_color),
         i(RenderOpts ^ ro_boundary_wall_width)], !IO).

    % Get the start point of a wall (clockwise direction).
    % This is the point we reach first when tracing the boundary clockwise.
    %
:- func wall_start_point(topology, bounds, int, int, boundary_wall) = string.

wall_start_point(square, _Bounds, S, M, boundary_wall(cell(X, Y), Dir)) = Str :-
    (
        Dir = left,
        % Left edge (going up): start is bottom-left corner of cell
        PX = float.float(M * S),
        PY = float.float((M + Y + 1) * S)
    ;
        Dir = up,
        % Top edge (going right): start is top-left corner of cell
        PX = float.float((M + X) * S),
        PY = float.float(M * S)
    ;
        Dir = right,
        % Right edge (going down): start is top-right corner of cell
        PX = float.float((M + X + 1) * S),
        PY = float.float((M + Y) * S)
    ;
        Dir = down,
        % Bottom edge (going left): start is bottom-right corner of cell
        PX = float.float((M + X + 1) * S),
        PY = float.float((M + Y + 1) * S)
    ;
        ( Dir = upper_left
        ; Dir = upper_right
        ; Dir = lower_left
        ; Dir = lower_right
        ),
        % Hex directions should not occur with square topology
        PX = 0.0,
        PY = 0.0
    ),
    Str = string.format("%s %s", [s(format_coord(PX)), s(format_coord(PY))]).

wall_start_point(hex, Bounds, S, M, boundary_wall(Cell, Dir)) = Str :-
    hex_cell_centre(S, M, Bounds, Cell, CX, CY),
    Sf = float.float(S),
    % Return the starting point of this edge when tracing the boundary clockwise.
    % The edge is identified by the cell and the outward direction.
    % Clockwise around the maze = counter-clockwise around each cell.
    hex_edge_start_point(Sf, CX, CY, Dir, PX, PY),
    Str = string.format("%s %s", [s(format_coord(PX)), s(format_coord(PY))]).

    % Get the starting point of a hex edge when tracing clockwise around the maze.
    % This is counter-clockwise around the cell itself.
    % Uses hex_edge_corners to determine the start corner.
    %
:- pred hex_edge_start_point(float::in, float::in, float::in, direction::in,
    float::out, float::out) is det.

hex_edge_start_point(Sf, CX, CY, Dir, PX, PY) :-
    hex_edge_corners(Dir, StartCorner, _EndCorner),
    hex_corner_offset(Sf, StartCorner, DX, DY),
    PX = CX + DX,
    PY = CY + DY.

    % Get the end point of a wall (clockwise direction).
    % This is the point we reach last when tracing the boundary clockwise.
    %
:- func wall_end_point(topology, bounds, int, int, boundary_wall) = string.

wall_end_point(square, _Bounds, S, M, boundary_wall(cell(X, Y), Dir)) = Str :-
    (
        Dir = left,
        % Left edge (going up): end is top-left corner of cell
        PX = float.float(M * S),
        PY = float.float((M + Y) * S)
    ;
        Dir = up,
        % Top edge (going right): end is top-right corner of cell
        PX = float.float((M + X + 1) * S),
        PY = float.float(M * S)
    ;
        Dir = right,
        % Right edge (going down): end is bottom-right corner of cell
        PX = float.float((M + X + 1) * S),
        PY = float.float((M + Y + 1) * S)
    ;
        Dir = down,
        % Bottom edge (going left): end is bottom-left corner of cell
        PX = float.float((M + X) * S),
        PY = float.float((M + Y + 1) * S)
    ;
        ( Dir = upper_left
        ; Dir = upper_right
        ; Dir = lower_left
        ; Dir = lower_right
        ),
        % Hex directions should not occur with square topology
        PX = 0.0,
        PY = 0.0
    ),
    Str = string.format("%s %s", [s(format_coord(PX)), s(format_coord(PY))]).

wall_end_point(hex, Bounds, S, M, boundary_wall(Cell, Dir)) = Str :-
    hex_cell_centre(S, M, Bounds, Cell, CX, CY),
    Sf = float.float(S),
    % Return the ending point of this edge when tracing the boundary clockwise.
    hex_edge_end_point(Sf, CX, CY, Dir, PX, PY),
    Str = string.format("%s %s", [s(format_coord(PX)), s(format_coord(PY))]).

    % Get the ending point of a hex edge when tracing clockwise around the maze.
    % This is counter-clockwise around the cell itself.
    % Uses hex_edge_corners to determine the end corner.
    %
:- pred hex_edge_end_point(float::in, float::in, float::in, direction::in,
    float::out, float::out) is det.

hex_edge_end_point(Sf, CX, CY, Dir, PX, PY) :-
    hex_edge_corners(Dir, _StartCorner, EndCorner),
    hex_corner_offset(Sf, EndCorner, DX, DY),
    PX = CX + DX,
    PY = CY + DY.

%---------------------------------------------------------------------------%

:- pred render_solution(io.text_output_stream::in, render_options::in,
    maze::in, list(cell)::in, io::di, io::uo) is det.

render_solution(Stream, RenderOpts, Maze, Solution, !IO) :-
    (
        Solution = []
    ;
        Solution = [_ | _],
        Topology = get_topology(Maze),
        Bounds = get_bounds(Maze),
        S = RenderOpts ^ ro_cell_spacing,
        M = RenderOpts ^ ro_margin,
        Points = list.map(cell_centre(Topology, Bounds, S, M), Solution),
        PointsStr = string.join_list(",", Points),
        io.format(Stream,
            "  <polyline points=\"%s\" fill=\"none\" stroke=\"%s\" stroke-width=\"%d\" stroke-linecap=\"round\" stroke-linejoin=\"round\"/>\n",
            [s(PointsStr), s(RenderOpts ^ ro_path_color),
             i(RenderOpts ^ ro_path_width)], !IO)
    ).

:- func cell_centre(topology, bounds, int, int, cell) = string.

cell_centre(square, _Bounds, S, M, cell(X, Y)) = Str :-
    Sf = float.float(S),
    Mf = float.float(M),
    CX = (Mf + float.float(X) + 0.5) * Sf,
    CY = (Mf + float.float(Y) + 0.5) * Sf,
    Str = string.format("%s %s", [s(format_coord(CX)), s(format_coord(CY))]).

cell_centre(hex, Bounds, S, M, Cell) = Str :-
    hex_cell_centre(S, M, Bounds, Cell, CX, CY),
    Str = string.format("%s %s", [s(format_coord(CX)), s(format_coord(CY))]).

%---------------------------------------------------------------------------%
:- end_module render.
%---------------------------------------------------------------------------%
