%---------------------------------------------------------------------------%
% vim: ft=mercury ts=4 sw=4 et
%---------------------------------------------------------------------------%
% File: options.m
%
% Command-line option parsing for the maze generator.
%
% 🤖 Generated with [Claude Code](https://claude.ai/code)
%
% Co-authored-by: Claude <noreply@anthropic.com>
%
%---------------------------------------------------------------------------%

:- module options.
:- interface.

:- import_module bool.
:- import_module io.
:- import_module list.
:- import_module maybe.

%---------------------------------------------------------------------------%

    % The grid topology.
    %
:- type topology
    --->    square
    ;       hex.

    % Command-line option identifiers.
    %
:- type option
    --->    help
    ;       topology_opt
    ;       size
    ;       random_seed
    ;       entrance_offset
    ;       exit_offset
    ;       algorithm
    ;       cell_spacing
    ;       margin
    ;       show_path
    ;       maze_bg_color
    ;       cell_bg_color
    ;       internal_wall_color
    ;       boundary_wall_color
    ;       path_color
    ;       internal_wall_width
    ;       boundary_wall_width
    ;       path_width
    ;       dump.

    % The maze generation algorithm.
    %
:- type algorithm
    --->    dfs
    ;       binary_tree
    ;       sidewinder
    ;       hunt_and_kill
    ;       aldous_broder
    ;       wilson
    ;       kruskal
    ;       prim.

    % Processed options in a convenient record form.
    %
:- type options
    --->    options(
                % Input options
                opt_topology :: topology,
                opt_width :: int,
                opt_height :: int,
                opt_entrance_offset :: int,
                opt_exit_offset :: int,
                opt_random_seed :: int,
                opt_algorithm :: algorithm,
                % Output options
                opt_cell_spacing :: int,
                opt_margin :: int,
                opt_show_path :: bool,
                opt_maze_bg_color :: string,
                opt_cell_bg_color :: string,
                opt_internal_wall_color :: string,
                opt_boundary_wall_color :: string,
                opt_path_color :: string,
                opt_internal_wall_width :: int,
                opt_boundary_wall_width :: int,
                opt_path_width :: int,
                % General options
                opt_dump :: bool,
                opt_output_file :: maybe(string)
            ).

    % Parse command-line arguments and return processed options.
    %
:- pred parse_options(list(string)::in, maybe(options)::out,
    io::di, io::uo) is det.

    % Display usage information.
    %
:- pred usage(io.text_output_stream::in, io::di, io::uo) is det.

%---------------------------------------------------------------------------%
%---------------------------------------------------------------------------%

:- implementation.

:- import_module char.
:- import_module getopt.
:- import_module int.
:- import_module string.
:- import_module time.

%---------------------------------------------------------------------------%

parse_options(Args, MaybeOptions, !IO) :-
    OptionOps = option_ops_multi(short_option, long_option, option_default),
    process_options(OptionOps, Args, NonOptionArgs, OptionResult),
    (
        OptionResult = ok(OptionTable),
        post_process_options(OptionTable, NonOptionArgs, MaybeOptions, !IO)
    ;
        OptionResult = error(Error),
        ErrorMsg = option_error_to_string(Error),
        io.stderr_stream(Stderr, !IO),
        io.format(Stderr, "Error: %s\n", [s(ErrorMsg)], !IO),
        MaybeOptions = no
    ).

:- pred post_process_options(option_table(option)::in, list(string)::in,
    maybe(options)::out, io::di, io::uo) is det.

post_process_options(OptionTable, NonOptionArgs, MaybeOptions, !IO) :-
    lookup_bool_option(OptionTable, help, Help),
    (
        Help = yes,
        io.stdout_stream(Stdout, !IO),
        usage(Stdout, !IO),
        MaybeOptions = no
    ;
        Help = no,
        % Input options
        lookup_maybe_string_option(OptionTable, topology_opt, MaybeTopologyStr),
        lookup_maybe_string_option(OptionTable, size, MaybeSizeStr),
        lookup_maybe_int_option(OptionTable, random_seed, MaybeSeed),
        lookup_maybe_int_option(OptionTable, entrance_offset, MaybeEntranceOffset),
        lookup_maybe_int_option(OptionTable, exit_offset, MaybeExitOffset),
        lookup_maybe_string_option(OptionTable, algorithm, MaybeAlgorithmStr),

        % Output options
        lookup_int_option(OptionTable, cell_spacing, CellSpacing),
        lookup_int_option(OptionTable, margin, Margin),
        lookup_bool_option(OptionTable, show_path, ShowPath),
        lookup_string_option(OptionTable, maze_bg_color, MazeBgColor),
        lookup_string_option(OptionTable, cell_bg_color, CellBgColor),
        lookup_string_option(OptionTable, internal_wall_color, InternalWallColor),
        lookup_string_option(OptionTable, boundary_wall_color, BoundaryWallColor),
        lookup_string_option(OptionTable, path_color, PathColor),
        lookup_int_option(OptionTable, internal_wall_width, InternalWallWidth),
        lookup_int_option(OptionTable, boundary_wall_width, BoundaryWallWidth),
        lookup_int_option(OptionTable, path_width, PathWidth),

        % General options
        lookup_bool_option(OptionTable, dump, Dump),

        % Get stderr for error messages
        io.stderr_stream(Stderr, !IO),

        % Process topology option (default: square)
        (
            MaybeTopologyStr = no,
            Topology = square,
            TopologyOk = yes
        ;
            MaybeTopologyStr = yes(TopologyStr),
            ( if parse_topology(TopologyStr, T) then
                Topology = T,
                TopologyOk = yes
            else
                io.format(Stderr, "Error: invalid topology '%s'\n",
                    [s(TopologyStr)], !IO),
                Topology = square,
                TopologyOk = no
            )
        ),

        % Process size option (default 10, interpreted based on topology)
        (
            MaybeSizeStr = no,
            ( if parse_size(Topology, "10", W0, H0) then
                Width = W0,
                Height = H0
            else
                % Should never fail for "10"
                Width = 10,
                Height = 10
            ),
            SizeOk = yes
        ;
            MaybeSizeStr = yes(SizeStr),
            ( if parse_size(Topology, SizeStr, W, H) then
                Width = W,
                Height = H,
                SizeOk = yes
            else
                io.format(Stderr, "Error: invalid size '%s'\n",
                    [s(SizeStr)], !IO),
                Width = 0,
                Height = 0,
                SizeOk = no
            )
        ),

        % Process entrance/exit offsets (default: 0 for entrance, same as entrance for exit)
        (
            MaybeEntranceOffset = yes(EntranceOffset)
        ;
            MaybeEntranceOffset = no,
            EntranceOffset = 0
        ),
        (
            MaybeExitOffset = yes(ExitOffset)
        ;
            MaybeExitOffset = no,
            ExitOffset = EntranceOffset
        ),

        % Get random seed (default from clock)
        (
            MaybeSeed = yes(Seed)
        ;
            MaybeSeed = no,
            time.clock(Seed, !IO)
        ),

        % Process algorithm option (default: kruskal)
        (
            MaybeAlgorithmStr = no,
            Algorithm = kruskal,
            AlgorithmOk = yes
        ;
            MaybeAlgorithmStr = yes(AlgorithmStr),
            ( if parse_algorithm(AlgorithmStr, A) then
                Algorithm = A,
                AlgorithmOk = yes
            else
                io.format(Stderr, "Error: invalid algorithm '%s'\n",
                    [s(AlgorithmStr)], !IO),
                Algorithm = dfs,
                AlgorithmOk = no
            )
        ),

        % Process output file
        (
            NonOptionArgs = [],
            OutputFile = no
        ;
            NonOptionArgs = [File],
            OutputFile = yes(File)
        ;
            NonOptionArgs = [_, _ | _],
            OutputFile = no
        ),

        % Check algorithm/topology compatibility
        ( if
            Topology = hex,
            (Algorithm = binary_tree ; Algorithm = sidewinder)
        then
            io.format(Stderr,
                "Error: algorithm '%s' is not compatible with hex topology\n",
                [s(algorithm_name(Algorithm))], !IO),
            CompatibilityOk = no
        else
            CompatibilityOk = yes
        ),

        ( if
            TopologyOk = yes,
            SizeOk = yes,
            AlgorithmOk = yes,
            CompatibilityOk = yes
        then
            Options = options(Topology, Width, Height, EntranceOffset,
                ExitOffset, Seed, Algorithm, CellSpacing, Margin, ShowPath,
                MazeBgColor, CellBgColor, InternalWallColor, BoundaryWallColor,
                PathColor, InternalWallWidth, BoundaryWallWidth, PathWidth,
                Dump, OutputFile),
            MaybeOptions = yes(Options)
        else
            MaybeOptions = no
        )
    ).

:- pred parse_topology(string::in, topology::out) is semidet.

parse_topology("square", square).
parse_topology("hex", hex).

:- pred parse_algorithm(string::in, algorithm::out) is semidet.

parse_algorithm("dfs", dfs).
parse_algorithm("binary-tree", binary_tree).
parse_algorithm("sidewinder", sidewinder).
parse_algorithm("hunt-and-kill", hunt_and_kill).
parse_algorithm("aldous-broder", aldous_broder).
parse_algorithm("wilson", wilson).
parse_algorithm("kruskal", kruskal).
parse_algorithm("prim", prim).

:- func algorithm_name(algorithm) = string.

algorithm_name(dfs) = "dfs".
algorithm_name(binary_tree) = "binary-tree".
algorithm_name(sidewinder) = "sidewinder".
algorithm_name(hunt_and_kill) = "hunt-and-kill".
algorithm_name(aldous_broder) = "aldous-broder".
algorithm_name(wilson) = "wilson".
algorithm_name(kruskal) = "kruskal".
algorithm_name(prim) = "prim".

:- pred parse_size(topology::in, string::in, int::out, int::out) is semidet.

parse_size(Topology, Str, Width, Height) :-
    ( if string.to_int(Str, N) then
        N > 0,
        (
            Topology = square,
            % For square grids: N means NxN
            Width = N,
            Height = N
        ;
            Topology = hex,
            % For hex grids: N means W = 2*N - 1, H = N (regular hexagon shape)
            Width = 2 * N - 1,
            Height = N
        )
    else if
        [WStr, HStr] = string.split_at_char('x', Str),
        string.to_int(WStr, W),
        string.to_int(HStr, H)
    then
        Width = W,
        Height = H,
        W > 0,
        H > 0
    else
        fail
    ).

%---------------------------------------------------------------------------%

:- pred short_option(char::in, option::out) is semidet.

short_option('h', help).
short_option('t', topology_opt).
short_option('s', size).
short_option('r', random_seed).
short_option('e', entrance_offset).
short_option('x', exit_offset).
short_option('a', algorithm).
short_option('p', show_path).
short_option('d', dump).

:- pred long_option(string::in, option::out) is semidet.

long_option("help", help).
long_option("topology", topology_opt).
long_option("size", size).
long_option("random-seed", random_seed).
long_option("entrance-offset", entrance_offset).
long_option("exit-offset", exit_offset).
long_option("algorithm", algorithm).
long_option("cell-spacing", cell_spacing).
long_option("margin", margin).
long_option("path", show_path).
long_option("maze-bg-color", maze_bg_color).
long_option("cell-bg-color", cell_bg_color).
long_option("internal-wall-color", internal_wall_color).
long_option("boundary-wall-color", boundary_wall_color).
long_option("path-color", path_color).
long_option("internal-wall-width", internal_wall_width).
long_option("boundary-wall-width", boundary_wall_width).
long_option("path-width", path_width).
long_option("dump", dump).

:- pred option_default(option::out, option_data::out) is multi.

option_default(help, bool(no)).
option_default(topology_opt, maybe_string(no)).
option_default(size, maybe_string(no)).
option_default(random_seed, maybe_int(no)).
option_default(entrance_offset, maybe_int(no)).
option_default(exit_offset, maybe_int(no)).
option_default(algorithm, maybe_string(no)).
option_default(cell_spacing, int(10)).
option_default(margin, int(2)).
option_default(show_path, bool(no)).
option_default(maze_bg_color, string("none")).
option_default(cell_bg_color, string("white")).
option_default(internal_wall_color, string("black")).
option_default(boundary_wall_color, string("black")).
option_default(path_color, string("red")).
option_default(internal_wall_width, int(1)).
option_default(boundary_wall_width, int(2)).
option_default(path_width, int(3)).
option_default(dump, bool(no)).

%---------------------------------------------------------------------------%

usage(Stream, !IO) :-
    io.write_string(Stream, "Usage: amaze [options] [output.svg]\n", !IO),
    io.write_string(Stream, "\n", !IO),
    io.write_string(Stream, "Input Options:\n", !IO),
    io.write_string(Stream, "  -t, --topology <square|hex>   Grid topology (default: square)\n", !IO),
    io.write_string(Stream, "  -s, --size <N|WxH>            Maze size (default: 10)\n", !IO),
    io.write_string(Stream, "  -r, --random-seed <N>         Random seed (default: system time)\n", !IO),
    io.write_string(Stream, "  -e, --entrance-offset <N>     Entrance offset from centre (default: 0)\n", !IO),
    io.write_string(Stream, "  -x, --exit-offset <N>         Exit offset from centre (default: entrance offset)\n", !IO),
    io.write_string(Stream, "  -a, --algorithm <name>        Maze generation algorithm (default: kruskal)\n", !IO),
    io.write_string(Stream, "                                Algorithms: dfs, binary-tree, sidewinder,\n", !IO),
    io.write_string(Stream, "                                hunt-and-kill, aldous-broder, wilson, kruskal, prim\n", !IO),
    io.write_string(Stream, "\n", !IO),
    io.write_string(Stream, "Output Options:\n", !IO),
    io.write_string(Stream, "  --cell-spacing <N>            Cell spacing in viewbox units (default: 10)\n", !IO),
    io.write_string(Stream, "  --margin <N>                  Margin in multiples of cell spacing (default: 2)\n", !IO),
    io.write_string(Stream, "  -p, --path                    Display the solution path\n", !IO),
    io.write_string(Stream, "  --maze-bg-color <color>       Maze background color (default: none)\n", !IO),
    io.write_string(Stream, "  --cell-bg-color <color>       Cell background color (default: white)\n", !IO),
    io.write_string(Stream, "  --internal-wall-color <color> Internal wall color (default: black)\n", !IO),
    io.write_string(Stream, "  --boundary-wall-color <color> Boundary wall color (default: black)\n", !IO),
    io.write_string(Stream, "  --path-color <color>          Solution path color (default: red)\n", !IO),
    io.write_string(Stream, "  --internal-wall-width <N>     Internal wall stroke width (default: 1)\n", !IO),
    io.write_string(Stream, "  --boundary-wall-width <N>     Boundary wall stroke width (default: 2)\n", !IO),
    io.write_string(Stream, "  --path-width <N>              Solution path stroke width (default: 3)\n", !IO),
    io.write_string(Stream, "\n", !IO),
    io.write_string(Stream, "General Options:\n", !IO),
    io.write_string(Stream, "  -h, --help                    Display this help\n", !IO),
    io.write_string(Stream, "  -d, --dump                    Dump maze to stdout\n", !IO),
    io.write_string(Stream, "\n", !IO),
    io.write_string(Stream, "Color values are passed through to SVG without validation.\n", !IO),
    io.write_string(Stream, "Valid formats: named colors (red), hex (#ff0000), rgb (rgb(255,0,0)).\n", !IO).

%---------------------------------------------------------------------------%
:- end_module options.
%---------------------------------------------------------------------------%
