%---------------------------------------------------------------------------%
% vim: ft=mercury ts=4 sw=4 et
%---------------------------------------------------------------------------%
% File: maze.m
%
% Maze data structure and operations.
%
% 🤖 Generated with [Claude Code](https://claude.ai/code)
%
% Co-authored-by: Claude <noreply@anthropic.com>
%
%---------------------------------------------------------------------------%

:- module maze.
:- interface.

:- import_module grid.
:- import_module list.
:- import_module options.
:- import_module set_tree234.

%---------------------------------------------------------------------------%

    % The dimensions of a rectangular maze.
    %
:- type bounds
    --->    bounds(
                width :: int,
                height :: int
            ).

    % The complete maze structure.
    %
:- type maze.

    % A boundary wall is an edge on the boundary of the maze, identified by
    % the cell inside the maze and the direction pointing outward.
    %
:- type boundary_wall
    --->    boundary_wall(cell, direction).

:- func wall_cell(boundary_wall) = cell.
:- func wall_direction(boundary_wall) = direction.

%---------------------------------------------------------------------------%

    % Create a maze with no internal doors (all walls).
    % The entrance and exit are specified as boundary walls.
    %
:- func init(topology, bounds, boundary_wall, boundary_wall) = maze.

    % Is this cell within the maze bounds?
    %
:- pred in_maze(maze::in, cell::in) is semidet.

    % Get the topology of the maze.
    %
:- func get_topology(maze) = topology.

    % Get the bounds of the maze.
    %
:- func get_bounds(maze) = bounds.

    % Get the entrance boundary wall.
    %
:- func get_entrance(maze) = boundary_wall.

    % Get the exit boundary wall.
    %
:- func get_exit(maze) = boundary_wall.

    % Get the start cell (cell at the entrance).
    %
:- func get_start_cell(maze) = cell.

    % Get the target cell (cell at the exit).
    %
:- func get_target_cell(maze) = cell.

    % Does this edge have a door?
    %
:- pred has_door(maze::in, edge::in) is semidet.

    % Add a door to an internal edge.
    %
:- func add_door(maze, edge) = maze.

    % List all cells in the maze.
    %
:- func cells(maze) = list(cell).

    % List all internal edges (edges between two cells both in the maze).
    %
:- func internal_edges(maze) = list(edge).

    % List adjacent cells reachable through doors.
    %
:- func neighbours(maze, cell) = list(cell).

    % List all adjacent cells that are within the maze bounds.
    %
:- func valid_neighbours(maze, cell) = list(cell).

    % List adjacent cells that are within the maze bounds and not in the
    % given set.
    %
:- func unvisited_neighbours(maze, set_tree234(cell), cell) = list(cell).

    % List adjacent cells that are within the maze bounds and in the
    % given set.
    %
:- func visited_neighbours(maze, set_tree234(cell), cell) = list(cell).

%---------------------------------------------------------------------------%
%---------------------------------------------------------------------------%

:- implementation.

:- import_module int.
:- import_module solutions.

%---------------------------------------------------------------------------%

:- type maze
    --->    maze(
                m_topology :: topology,
                m_bounds :: bounds,
                m_doors :: set_tree234(edge),
                m_entrance :: boundary_wall,
                m_exit :: boundary_wall
            ).

%---------------------------------------------------------------------------%

wall_cell(boundary_wall(Cell, _)) = Cell.
wall_direction(boundary_wall(_, Dir)) = Dir.

%---------------------------------------------------------------------------%

init(Topology, Bounds, Entrance, Exit) = Maze :-
    Doors = set_tree234.init,
    Maze = maze(Topology, Bounds, Doors, Entrance, Exit).

%---------------------------------------------------------------------------%

in_maze(Maze, Cell) :-
    Topology = Maze ^ m_topology,
    Bounds = Maze ^ m_bounds,
    (
        Topology = square,
        in_maze_square(Bounds, Cell)
    ;
        Topology = hex,
        in_maze_hex(Bounds, Cell)
    ).

:- pred in_maze_square(bounds::in, cell::in) is semidet.

in_maze_square(Bounds, cell(X, Y)) :-
    X >= 0, X < Bounds ^ width,
    Y >= 0, Y < Bounds ^ height.

:- pred in_maze_hex(bounds::in, cell::in) is semidet.

in_maze_hex(Bounds, cell(Q, R)) :-
    W = Bounds ^ width,
    H = Bounds ^ height,
    HalfW = (W - 1) / 2,
    % Q in [0, W)
    Q >= 0, Q < W,
    % Top constraint: for Q < HalfW, R >= 0; for Q >= HalfW, R >= Q - HalfW
    % Combined as: R >= max(0, Q - HalfW)
    R >= int.max(0, Q - HalfW),
    % Bottom constraint: for Q <= HalfW, R <= Q + H - 1; for Q > HalfW, R <= HalfW + H - 1
    % Combined as: R <= min(Q + H - 1, HalfW + H - 1)
    R =< int.min(Q + H - 1, HalfW + H - 1).

%---------------------------------------------------------------------------%

get_topology(Maze) = Maze ^ m_topology.
get_bounds(Maze) = Maze ^ m_bounds.
get_entrance(Maze) = Maze ^ m_entrance.
get_exit(Maze) = Maze ^ m_exit.
get_start_cell(Maze) = wall_cell(Maze ^ m_entrance).
get_target_cell(Maze) = wall_cell(Maze ^ m_exit).

%---------------------------------------------------------------------------%

has_door(Maze, Edge) :-
    set_tree234.member(Edge, Maze ^ m_doors).

add_door(Maze0, Edge) = Maze :-
    Doors0 = Maze0 ^ m_doors,
    set_tree234.insert(Edge, Doors0, Doors),
    Maze = Maze0 ^ m_doors := Doors.

%---------------------------------------------------------------------------%

cells(Maze) = Cells :-
    Topology = Maze ^ m_topology,
    Bounds = Maze ^ m_bounds,
    (
        Topology = square,
        Cells = cells_square(Bounds)
    ;
        Topology = hex,
        Cells = cells_hex(Bounds)
    ).

:- func cells_square(bounds) = list(cell).

cells_square(Bounds) = Cells :-
    Width = Bounds ^ width,
    Height = Bounds ^ height,
    solutions(
        ( pred(Cell::out) is nondet :-
            int.nondet_int_in_range(0, Width - 1, X),
            int.nondet_int_in_range(0, Height - 1, Y),
            Cell = cell(X, Y)
        ),
        Cells).

:- func cells_hex(bounds) = list(cell).

cells_hex(Bounds) = Cells :-
    W = Bounds ^ width,
    H = Bounds ^ height,
    HalfW = (W - 1) / 2,
    % R ranges from 0 to HalfW + H - 1
    MaxR = HalfW + H - 1,
    solutions(
        ( pred(Cell::out) is nondet :-
            int.nondet_int_in_range(0, W - 1, Q),
            int.nondet_int_in_range(0, MaxR, R),
            % Apply constraints matching in_maze_hex
            R >= int.max(0, Q - HalfW),
            R =< int.min(Q + H - 1, HalfW + H - 1),
            Cell = cell(Q, R)
        ),
        Cells).

%---------------------------------------------------------------------------%

internal_edges(Maze) = Edges :-
    Topology = Maze ^ m_topology,
    CanonDirs = canonical_directions(Topology),
    AllCells = cells(Maze),
    solutions(
        ( pred(Edge::out) is nondet :-
            list.member(Cell, AllCells),
            list.member(Dir, CanonDirs),
            AdjCell = adjacent(Topology, Cell, Dir),
            in_maze(Maze, AdjCell),
            Edge = get_edge(Topology, Cell, Dir)
        ),
        Edges).

%---------------------------------------------------------------------------%

neighbours(Maze, Cell) = Neighbours :-
    Topology = Maze ^ m_topology,
    Dirs = directions(Topology),
    solutions(
        ( pred(Neighbour::out) is nondet :-
            list.member(Dir, Dirs),
            Neighbour = adjacent(Topology, Cell, Dir),
            in_maze(Maze, Neighbour),
            Edge = get_edge(Topology, Cell, Dir),
            has_door(Maze, Edge)
        ),
        Neighbours).

%---------------------------------------------------------------------------%

valid_neighbours(Maze, Cell) = Neighbours :-
    Topology = Maze ^ m_topology,
    Dirs = directions(Topology),
    list.filter_map(
        ( pred(Dir::in, Neighbour::out) is semidet :-
            Neighbour = adjacent(Topology, Cell, Dir),
            in_maze(Maze, Neighbour)
        ),
        Dirs,
        Neighbours).

unvisited_neighbours(Maze, Visited, Cell) = Unvisited :-
    Topology = Maze ^ m_topology,
    Dirs = directions(Topology),
    list.filter_map(
        ( pred(Dir::in, Neighbour::out) is semidet :-
            Neighbour = adjacent(Topology, Cell, Dir),
            in_maze(Maze, Neighbour),
            not set_tree234.member(Neighbour, Visited)
        ),
        Dirs,
        Unvisited).

visited_neighbours(Maze, Visited, Cell) = VisitedNbrs :-
    Topology = Maze ^ m_topology,
    Dirs = directions(Topology),
    list.filter_map(
        ( pred(Dir::in, Neighbour::out) is semidet :-
            Neighbour = adjacent(Topology, Cell, Dir),
            in_maze(Maze, Neighbour),
            set_tree234.member(Neighbour, Visited)
        ),
        Dirs,
        VisitedNbrs).

%---------------------------------------------------------------------------%
:- end_module maze.
%---------------------------------------------------------------------------%
