%---------------------------------------------------------------------------%
% vim: ft=mercury ts=4 sw=4 et
%---------------------------------------------------------------------------%
% File: generate.wilson.m
%
% Wilson's maze generation algorithm.
%
% 🤖 Generated with [Claude Code](https://claude.ai/code)
%
% Co-authored-by: Claude <noreply@anthropic.com>
%
%---------------------------------------------------------------------------%

:- module generate.wilson.
:- interface.

:- import_module maze.
:- import_module random.
:- import_module random.sfc16.

%---------------------------------------------------------------------------%

    % Generate a maze using Wilson's algorithm.
    %
    % 1. Mark a random cell as part of the maze.
    % 2. While there are cells not in the maze:
    %    a. Start a random walk from a random unvisited cell.
    %    b. Walk randomly until hitting a cell that's already in the maze,
    %       erasing any loops formed during the walk.
    %    c. Add the loop-erased path to the maze, carving doors along it.
    %
    % This produces uniformly random spanning trees like Aldous-Broder,
    % but more efficiently due to loop-erasing.
    %
:- pred generate_wilson(maze::in, maze::out,
    random.sfc16.random::in, random.sfc16.random::out) is det.

%---------------------------------------------------------------------------%
%---------------------------------------------------------------------------%

:- implementation.

:- import_module generate.
:- import_module grid.
:- import_module options.

:- import_module int.
:- import_module list.
:- import_module map.
:- import_module require.
:- import_module set_tree234.

%---------------------------------------------------------------------------%

generate_wilson(!Maze, !RNG) :-
    % Start with the start cell in the maze
    StartCell = get_start_cell(!.Maze),
    InMaze0 = set_tree234.init,
    set_tree234.insert(StartCell, InMaze0, InMaze1),
    TotalCells = count_cells(!.Maze),
    wilson_loop(1, TotalCells, InMaze1, _, !Maze, !RNG).

    % Count total cells in the maze.
    %
:- func count_cells(maze) = int.

count_cells(Maze) = list.length(cells(Maze)).

    % Main Wilson's loop - keep adding paths until all cells are in the maze.
    %
:- pred wilson_loop(int::in, int::in,
    set_tree234(cell)::in, set_tree234(cell)::out,
    maze::in, maze::out,
    random.sfc16.random::in, random.sfc16.random::out) is det.

wilson_loop(InMazeCount, TotalCells, !InMaze, !Maze, !RNG) :-
    ( if InMazeCount >= TotalCells then
        % All cells in maze - done
        true
    else
        % Find a cell not in the maze to start the walk
        find_cell_not_in_maze(!.Maze, !.InMaze, StartCell, !RNG),
        % Do a loop-erased random walk until we hit the maze
        loop_erased_walk(!.Maze, !.InMaze, StartCell, map.init, Path, !RNG),
        % Add the path to the maze
        add_path_to_maze(Path, AddedCount, !InMaze, !Maze),
        wilson_loop(InMazeCount + AddedCount, TotalCells, !InMaze, !Maze, !RNG)
    ).

    % Find a random cell that is not in the maze.
    %
:- pred find_cell_not_in_maze(maze::in, set_tree234(cell)::in, cell::out,
    random.sfc16.random::in, random.sfc16.random::out) is det.

find_cell_not_in_maze(Maze, InMaze, Cell, !RNG) :-
    AllCells = cells(Maze),
    NotInMaze = list.filter(
        ( pred(C::in) is semidet :- not set_tree234.member(C, InMaze) ),
        AllCells),
    (
        NotInMaze = [],
        unexpected($pred, "no cells outside maze")
    ;
        NotInMaze = [_ | _],
        choose_random(NotInMaze, Cell, !RNG)
    ).

    % Perform a loop-erased random walk from StartCell until hitting a cell
    % in the maze. Returns the path as a map from each cell to the next cell.
    %
:- pred loop_erased_walk(maze::in, set_tree234(cell)::in, cell::in,
    map(cell, cell)::in, map(cell, cell)::out,
    random.sfc16.random::in, random.sfc16.random::out) is det.

loop_erased_walk(Maze, InMaze, Current, !Path, !RNG) :-
    ( if set_tree234.member(Current, InMaze) then
        % Reached the maze - done
        true
    else
        % Choose a random neighbour
        Neighbours = valid_neighbours(Maze, Current),
        choose_random(Neighbours, Next, !RNG),
        % If we've visited Current before, this creates a loop.
        % By updating the map, we effectively erase the loop.
        map.set(Current, Next, !Path),
        loop_erased_walk(Maze, InMaze, Next, !Path, !RNG)
    ).

    % Add the path to the maze by following the map from cells not in maze
    % until we reach a cell in the maze.
    %
:- pred add_path_to_maze(map(cell, cell)::in, int::out,
    set_tree234(cell)::in, set_tree234(cell)::out,
    maze::in, maze::out) is det.

add_path_to_maze(Path, AddedCount, !InMaze, !Maze) :-
    % Find a starting cell (any key in the path that's not in the maze)
    map.keys(Path, PathCells),
    StartCells = list.filter(
        ( pred(C::in) is semidet :- not set_tree234.member(C, !.InMaze) ),
        PathCells),
    (
        StartCells = [],
        % No cells to add (shouldn't happen)
        AddedCount = 0
    ;
        StartCells = [Start | _],
        follow_path(Path, Start, 0, AddedCount, !InMaze, !Maze)
    ).

    % Follow the path from Current, adding cells to the maze and carving doors.
    %
:- pred follow_path(map(cell, cell)::in, cell::in, int::in, int::out,
    set_tree234(cell)::in, set_tree234(cell)::out,
    maze::in, maze::out) is det.

follow_path(Path, Current, !AddedCount, !InMaze, !Maze) :-
    ( if set_tree234.member(Current, !.InMaze) then
        % Reached the maze - done
        true
    else
        % Add current to maze
        set_tree234.insert(Current, !InMaze),
        !:AddedCount = !.AddedCount + 1,
        % Get next cell and carve door
        ( if map.search(Path, Current, Next) then
            Topology = get_topology(!.Maze),
            Edge = get_edge_between(Topology, Current, Next),
            !:Maze = add_door(!.Maze, Edge),
            follow_path(Path, Next, !AddedCount, !InMaze, !Maze)
        else
            unexpected($pred, "path ends unexpectedly")
        )
    ).

%---------------------------------------------------------------------------%
:- end_module generate.wilson.
%---------------------------------------------------------------------------%
