%---------------------------------------------------------------------------%
% vim: ft=mercury ts=4 sw=4 et
%---------------------------------------------------------------------------%
% File: generate.sidewinder.m
%
% Sidewinder maze generation algorithm.
%
% 🤖 Generated with [Claude Code](https://claude.ai/code)
%
% Co-authored-by: Claude <noreply@anthropic.com>
%
%---------------------------------------------------------------------------%

:- module generate.sidewinder.
:- interface.

:- import_module maze.
:- import_module random.
:- import_module random.sfc16.

%---------------------------------------------------------------------------%

    % Generate a maze using the Sidewinder algorithm.
    %
    % For each row (bottom to top):
    %   - For each cell, randomly decide to carve east or close the run.
    %   - When closing a run, pick a random cell from the run and carve up.
    %   - At the east boundary, always close the run.
    % The top row always carves east (no up option).
    %
    % This produces a maze with a clear path along the top edge.
    %
:- pred generate_sidewinder(maze::in, maze::out,
    random.sfc16.random::in, random.sfc16.random::out) is det.

%---------------------------------------------------------------------------%
%---------------------------------------------------------------------------%

:- implementation.

:- import_module grid.
:- import_module options.

:- import_module bool.
:- import_module int.
:- import_module list.

%---------------------------------------------------------------------------%

generate_sidewinder(!Maze, !RNG) :-
    Bounds = get_bounds(!.Maze),
    Width = Bounds ^ width,
    Height = Bounds ^ height,
    % Process rows from bottom (0) to top (Height-1)
    % Top row (Height-1) is special: always carve east
    process_rows(Width, Height, 0, !Maze, !RNG).

:- pred process_rows(int::in, int::in, int::in, maze::in, maze::out,
    random.sfc16.random::in, random.sfc16.random::out) is det.

process_rows(Width, Height, Y, !Maze, !RNG) :-
    ( if Y >= Height then
        true
    else
        process_row(Width, Height, Y, !Maze, !RNG),
        process_rows(Width, Height, Y + 1, !Maze, !RNG)
    ).

    % Process a single row.
    %
:- pred process_row(int::in, int::in, int::in, maze::in, maze::out,
    random.sfc16.random::in, random.sfc16.random::out) is det.

process_row(Width, Height, Y, !Maze, !RNG) :-
    % Start with empty run at X=0
    process_row_cells(Width, Height, Y, 0, [], !Maze, !RNG).

    % Process cells in a row, maintaining the current run.
    % Run is the list of cells in the current run (in reverse order).
    %
:- pred process_row_cells(int::in, int::in, int::in, int::in, list(cell)::in,
    maze::in, maze::out,
    random.sfc16.random::in, random.sfc16.random::out) is det.

process_row_cells(Width, Height, Y, X, Run, !Maze, !RNG) :-
    ( if X >= Width then
        % End of row
        true
    else
        Cell = cell(X, Y),
        NewRun = [Cell | Run],
        ( if X = Width - 1 then
            AtEastBoundary = yes
        else
            AtEastBoundary = no
        ),
        ( if Y = Height - 1 then
            AtTopRow = yes
        else
            AtTopRow = no
        ),
        ( if AtTopRow = yes then
            % Top row: always carve east if possible
            ( if AtEastBoundary = yes then
                % Top-right corner: done
                true
            else
                % Carve east
                carve_door(Cell, right, !Maze),
                process_row_cells(Width, Height, Y, X + 1, NewRun, !Maze, !RNG)
            )
        else if AtEastBoundary = yes then
            % East boundary: must close the run
            close_run(NewRun, !Maze, !RNG),
            process_row_cells(Width, Height, Y, X + 1, [], !Maze, !RNG)
        else
            % Interior cell: randomly carve east or close run
            random.uniform_int_in_range(0, 2, Choice, !RNG),
            ( if Choice = 0 then
                % Close the run
                close_run(NewRun, !Maze, !RNG),
                process_row_cells(Width, Height, Y, X + 1, [], !Maze, !RNG)
            else
                % Carve east and continue run
                carve_door(Cell, right, !Maze),
                process_row_cells(Width, Height, Y, X + 1, NewRun, !Maze, !RNG)
            )
        )
    ).

    % Close a run by picking a random cell and carving up from it.
    %
:- pred close_run(list(cell)::in, maze::in, maze::out,
    random.sfc16.random::in, random.sfc16.random::out) is det.

close_run(Run, !Maze, !RNG) :-
    (
        Run = []
        % Empty run, nothing to do (shouldn't happen)
    ;
        Run = [_ | _],
        % Pick a random cell from the run and carve up
        Length = list.length(Run),
        random.uniform_int_in_range(0, Length, Index, !RNG),
        list.det_index0(Run, Index, ChosenCell),
        carve_door(ChosenCell, up, !Maze)
    ).

    % Carve a door from a cell in the given direction.
    %
:- pred carve_door(cell::in, direction::in, maze::in, maze::out) is det.

carve_door(Cell, Dir, !Maze) :-
    Topology = get_topology(!.Maze),
    Edge = get_edge(Topology, Cell, Dir),
    !:Maze = add_door(!.Maze, Edge).

%---------------------------------------------------------------------------%
:- end_module generate.sidewinder.
%---------------------------------------------------------------------------%
