%---------------------------------------------------------------------------%
% vim: ft=mercury ts=4 sw=4 et
%---------------------------------------------------------------------------%
% File: generate.prim.m
%
% Prim's maze generation algorithm.
%
% 🤖 Generated with [Claude Code](https://claude.ai/code)
%
% Co-authored-by: Claude <noreply@anthropic.com>
%
%---------------------------------------------------------------------------%

:- module generate.prim.
:- interface.

:- import_module maze.
:- import_module random.
:- import_module random.sfc16.

%---------------------------------------------------------------------------%

    % Generate a maze using Prim's algorithm.
    %
    % 1. Start with the start cell in the maze.
    % 2. Add all neighbours of the start cell to the frontier.
    % 3. While the frontier is not empty:
    %    a. Pick a random cell from the frontier.
    %    b. Find its neighbours that are already in the maze.
    %    c. Pick one randomly and carve a door between them.
    %    d. Add the frontier cell to the maze.
    %    e. Add its unvisited neighbours to the frontier.
    %
    % The maze grows outward from the starting point.
    %
:- pred generate_prim(maze::in, maze::out,
    random.sfc16.random::in, random.sfc16.random::out) is det.

%---------------------------------------------------------------------------%
%---------------------------------------------------------------------------%

:- implementation.

:- import_module generate.
:- import_module grid.
:- import_module options.

:- import_module list.
:- import_module set_tree234.

%---------------------------------------------------------------------------%

generate_prim(!Maze, !RNG) :-
    % Start with the start cell in the maze
    StartCell = get_start_cell(!.Maze),
    InMaze0 = set_tree234.init,
    set_tree234.insert(StartCell, InMaze0, InMaze1),

    % Initialize frontier with neighbours of start cell
    InitialNeighbours = valid_neighbours(!.Maze, StartCell),
    Frontier0 = set_tree234.list_to_set(InitialNeighbours),

    % Main loop
    prim_loop(InMaze1, _, Frontier0, _, !Maze, !RNG).

    % Main Prim's loop - keep adding frontier cells until done.
    %
:- pred prim_loop(set_tree234(cell)::in, set_tree234(cell)::out,
    set_tree234(cell)::in, set_tree234(cell)::out,
    maze::in, maze::out,
    random.sfc16.random::in, random.sfc16.random::out) is det.

prim_loop(!InMaze, !Frontier, !Maze, !RNG) :-
    ( if set_tree234.is_empty(!.Frontier) then
        % Done - frontier is empty
        true
    else
        % Pick a random cell from the frontier
        FrontierList = set_tree234.to_sorted_list(!.Frontier),
        choose_random(FrontierList, ChosenCell, !RNG),

        % Remove chosen cell from frontier
        set_tree234.delete(ChosenCell, !Frontier),

        % Find neighbours of chosen cell that are in the maze
        InMazeNeighbours = visited_neighbours(!.Maze, !.InMaze, ChosenCell),

        % Pick one randomly and carve a door
        choose_random(InMazeNeighbours, ConnectTo, !RNG),
        Topology = get_topology(!.Maze),
        Edge = get_edge_between(Topology, ChosenCell, ConnectTo),
        !:Maze = add_door(!.Maze, Edge),

        % Add chosen cell to the maze
        set_tree234.insert(ChosenCell, !InMaze),

        % Add unvisited neighbours to frontier
        add_unvisited_to_frontier(!.Maze, !.InMaze, ChosenCell, !Frontier),

        % Continue
        prim_loop(!InMaze, !Frontier, !Maze, !RNG)
    ).

    % Add unvisited neighbours of a cell to the frontier.
    %
:- pred add_unvisited_to_frontier(maze::in, set_tree234(cell)::in, cell::in,
    set_tree234(cell)::in, set_tree234(cell)::out) is det.

add_unvisited_to_frontier(Maze, InMaze, Cell, !Frontier) :-
    Neighbours = valid_neighbours(Maze, Cell),
    list.foldl(
        ( pred(N::in, F0::in, F::out) is det :-
            ( if set_tree234.member(N, InMaze) then
                % Already in maze, skip
                F = F0
            else
                % Add to frontier (set handles duplicates)
                set_tree234.insert(N, F0, F)
            )
        ),
        Neighbours,
        !Frontier).

%---------------------------------------------------------------------------%
:- end_module generate.prim.
%---------------------------------------------------------------------------%
