%---------------------------------------------------------------------------%
% vim: ft=mercury ts=4 sw=4 et
%---------------------------------------------------------------------------%
% File: generate.m
%
% Maze generation algorithms.
%
% 🤖 Generated with [Claude Code](https://claude.ai/code)
%
% Co-authored-by: Claude <noreply@anthropic.com>
%
%---------------------------------------------------------------------------%

:- module generate.
:- interface.

:- include_module aldous_broder.
:- include_module binary_tree.
:- include_module dfs.
:- include_module hunt_and_kill.
:- include_module kruskal.
:- include_module prim.
:- include_module sidewinder.
:- include_module wilson.

:- import_module list.
:- import_module maze.
:- import_module options.
:- import_module random.
:- import_module random.sfc16.

%---------------------------------------------------------------------------%

    % Generate a maze by carving doors into an empty maze.
    %
:- pred generate(algorithm::in, maze::in, maze::out,
    random.sfc16.random::in, random.sfc16.random::out) is det.

    % Choose a random element from a non-empty list.
    % Throws an exception if the list is empty.
    %
:- pred choose_random(list(T)::in, T::out,
    random.sfc16.random::in, random.sfc16.random::out) is det.

%---------------------------------------------------------------------------%
%---------------------------------------------------------------------------%

:- implementation.

:- import_module generate.aldous_broder.
:- import_module generate.binary_tree.
:- import_module generate.dfs.
:- import_module generate.hunt_and_kill.
:- import_module generate.kruskal.
:- import_module generate.prim.
:- import_module generate.sidewinder.
:- import_module generate.wilson.

:- import_module int.
:- import_module require.

%---------------------------------------------------------------------------%

generate(Algorithm, !Maze, !RNG) :-
    (
        Algorithm = dfs,
        generate_dfs(!Maze, !RNG)
    ;
        Algorithm = binary_tree,
        generate_binary_tree(!Maze, !RNG)
    ;
        Algorithm = sidewinder,
        generate_sidewinder(!Maze, !RNG)
    ;
        Algorithm = hunt_and_kill,
        generate_hunt_and_kill(!Maze, !RNG)
    ;
        Algorithm = aldous_broder,
        generate_aldous_broder(!Maze, !RNG)
    ;
        Algorithm = wilson,
        generate_wilson(!Maze, !RNG)
    ;
        Algorithm = kruskal,
        generate_kruskal(!Maze, !RNG)
    ;
        Algorithm = prim,
        generate_prim(!Maze, !RNG)
    ).

%---------------------------------------------------------------------------%

choose_random(List, Chosen, !RNG) :-
    (
        List = [],
        unexpected($pred, "empty list")
    ;
        List = [_ | _],
        Length = list.length(List),
        random.uniform_int_in_range(0, Length, Index, !RNG),
        list.det_index0(List, Index, Chosen)
    ).

%---------------------------------------------------------------------------%
:- end_module generate.
%---------------------------------------------------------------------------%
