%---------------------------------------------------------------------------%
% vim: ft=mercury ts=4 sw=4 et
%---------------------------------------------------------------------------%
% File: generate.kruskal.m
%
% Kruskal's maze generation algorithm.
%
% 🤖 Generated with [Claude Code](https://claude.ai/code)
%
% Co-authored-by: Claude <noreply@anthropic.com>
%
%---------------------------------------------------------------------------%

:- module generate.kruskal.
:- interface.

:- import_module maze.
:- import_module random.
:- import_module random.sfc16.

%---------------------------------------------------------------------------%

    % Generate a maze using Kruskal's algorithm.
    %
    % 1. Create a list of all internal edges (potential walls).
    % 2. Shuffle the list randomly.
    % 3. For each edge: if the cells on either side are not yet connected,
    %    add a door and merge their sets.
    %
    % Uses a union-find data structure to track connected components.
    %
:- pred generate_kruskal(maze::in, maze::out,
    random.sfc16.random::in, random.sfc16.random::out) is det.

%---------------------------------------------------------------------------%
%---------------------------------------------------------------------------%

:- implementation.

:- import_module grid.
:- import_module options.

:- import_module int.
:- import_module list.
:- import_module map.

%---------------------------------------------------------------------------%

% Union-Find data structure
% Each cell maps to its parent. A cell that maps to itself is a root.

:- type union_find == map(cell, cell).

    % Initialize union-find with each cell as its own set.
    %
:- func init_union_find(list(cell)) = union_find.

init_union_find(Cells) = UF :-
    list.foldl(
        ( pred(C::in, M0::in, M::out) is det :- map.det_insert(C, C, M0, M) ),
        Cells,
        map.init,
        UF).

    % Find the representative (root) of the set containing Cell.
    %
:- func find(union_find, cell) = cell.

find(UF, Cell) = Root :-
    map.lookup(UF, Cell, Parent),
    ( if Parent = Cell then
        Root = Cell
    else
        Root = find(UF, Parent)
    ).

    % Union the sets containing Cell1 and Cell2.
    % Returns the updated union-find and whether the cells were in different sets.
    %
:- pred union(cell::in, cell::in, union_find::in, union_find::out,
    bool::out) is det.

:- import_module bool.

union(Cell1, Cell2, !UF, Merged) :-
    Root1 = find(!.UF, Cell1),
    Root2 = find(!.UF, Cell2),
    ( if Root1 = Root2 then
        % Already in the same set
        Merged = no
    else
        % Merge by making Root1 point to Root2
        map.det_update(Root1, Root2, !UF),
        Merged = yes
    ).

%---------------------------------------------------------------------------%

generate_kruskal(!Maze, !RNG) :-
    % Get all cells and internal edges
    AllCells = cells(!.Maze),
    AllEdges = internal_edges(!.Maze),

    % Initialize union-find
    UF0 = init_union_find(AllCells),

    % Shuffle the edges
    shuffle(AllEdges, ShuffledEdges, !RNG),

    % Process each edge
    Topology = get_topology(!.Maze),
    process_edges(Topology, ShuffledEdges, UF0, _, !Maze).

    % Process edges, adding doors where cells are not yet connected.
    %
:- pred process_edges(topology::in, list(edge)::in,
    union_find::in, union_find::out,
    maze::in, maze::out) is det.

process_edges(_, [], !UF, !Maze).
process_edges(Topology, [Edge | Edges], !UF, !Maze) :-
    edge_cells(Topology, Edge, Cell1, Cell2),
    union(Cell1, Cell2, !UF, Merged),
    (
        Merged = yes,
        !:Maze = add_door(!.Maze, Edge)
    ;
        Merged = no
    ),
    process_edges(Topology, Edges, !UF, !Maze).

%---------------------------------------------------------------------------%

    % Shuffle a list using Fisher-Yates algorithm.
    %
:- pred shuffle(list(T)::in, list(T)::out,
    random.sfc16.random::in, random.sfc16.random::out) is det.

shuffle(List, Shuffled, !RNG) :-
    array_shuffle(List, Shuffled, !RNG).

    % Fisher-Yates shuffle using list operations.
    % Not the most efficient but simple and correct.
    %
:- pred array_shuffle(list(T)::in, list(T)::out,
    random.sfc16.random::in, random.sfc16.random::out) is det.

array_shuffle(List, Shuffled, !RNG) :-
    Length = list.length(List),
    do_shuffle(Length, List, Shuffled, !RNG).

:- pred do_shuffle(int::in, list(T)::in, list(T)::out,
    random.sfc16.random::in, random.sfc16.random::out) is det.

do_shuffle(Remaining, !List, !RNG) :-
    ( if Remaining =< 1 then
        true
    else
        % Pick random index from 0 to Remaining-1
        random.uniform_int_in_range(0, Remaining, Index, !RNG),
        % Swap element at Index with element at Remaining-1
        swap_elements(Index, Remaining - 1, !List),
        do_shuffle(Remaining - 1, !List, !RNG)
    ).

    % Swap elements at positions I and J in a list.
    %
:- pred swap_elements(int::in, int::in, list(T)::in, list(T)::out) is det.

swap_elements(I, J, List0, List) :-
    ( if I = J then
        List = List0
    else
        list.det_index0(List0, I, ElemI),
        list.det_index0(List0, J, ElemJ),
        list.det_replace_nth(List0, I + 1, ElemJ, List1),
        list.det_replace_nth(List1, J + 1, ElemI, List)
    ).

%---------------------------------------------------------------------------%
:- end_module generate.kruskal.
%---------------------------------------------------------------------------%
