%---------------------------------------------------------------------------%
% vim: ft=mercury ts=4 sw=4 et
%---------------------------------------------------------------------------%
% File: generate.hunt_and_kill.m
%
% Hunt-and-Kill maze generation algorithm.
%
% 🤖 Generated with [Claude Code](https://claude.ai/code)
%
% Co-authored-by: Claude <noreply@anthropic.com>
%
%---------------------------------------------------------------------------%

:- module generate.hunt_and_kill.
:- interface.

:- import_module maze.
:- import_module random.
:- import_module random.sfc16.

%---------------------------------------------------------------------------%

    % Generate a maze using the Hunt-and-Kill algorithm.
    %
    % 1. Start from a random cell and mark it as visited.
    % 2. Perform a random walk, carving doors to unvisited neighbours.
    % 3. When stuck (no unvisited neighbours):
    %    a. Scan the grid for an unvisited cell adjacent to a visited cell.
    %    b. If found, carve a door between them and continue the walk.
    %    c. If not found, the maze is complete.
    %
:- pred generate_hunt_and_kill(maze::in, maze::out,
    random.sfc16.random::in, random.sfc16.random::out) is det.

%---------------------------------------------------------------------------%
%---------------------------------------------------------------------------%

:- implementation.

:- import_module generate.
:- import_module grid.
:- import_module options.

:- import_module list.
:- import_module maybe.
:- import_module set_tree234.

%---------------------------------------------------------------------------%

generate_hunt_and_kill(!Maze, !RNG) :-
    StartCell = get_start_cell(!.Maze),
    Visited0 = set_tree234.init,
    set_tree234.insert(StartCell, Visited0, Visited1),
    kill_phase(StartCell, Visited1, _, !Maze, !RNG).

    % Kill phase: random walk from current cell, carving doors to unvisited
    % neighbours until we hit a dead end.
    %
:- pred kill_phase(cell::in, set_tree234(cell)::in, set_tree234(cell)::out,
    maze::in, maze::out,
    random.sfc16.random::in, random.sfc16.random::out) is det.

kill_phase(Current, !Visited, !Maze, !RNG) :-
    Unvisited = unvisited_neighbours(!.Maze, !.Visited, Current),
    (
        Unvisited = [_ | _],
        % Choose a random unvisited neighbour
        choose_random(Unvisited, Chosen, !RNG),
        % Carve a door between current and chosen
        Topology = get_topology(!.Maze),
        Edge = get_edge_between(Topology, Current, Chosen),
        !:Maze = add_door(!.Maze, Edge),
        % Mark chosen as visited
        set_tree234.insert(Chosen, !Visited),
        % Continue walking from chosen
        kill_phase(Chosen, !Visited, !Maze, !RNG)
    ;
        Unvisited = [],
        % Dead end - enter hunt phase
        hunt_phase(!Visited, !Maze, !RNG)
    ).

    % Hunt phase: scan the grid for an unvisited cell that is adjacent to
    % a visited cell. If found, carve a door and resume the kill phase.
    %
:- pred hunt_phase(set_tree234(cell)::in, set_tree234(cell)::out,
    maze::in, maze::out,
    random.sfc16.random::in, random.sfc16.random::out) is det.

hunt_phase(!Visited, !Maze, !RNG) :-
    find_hunt_target(!.Maze, !.Visited, MaybeTarget),
    (
        MaybeTarget = yes({UnvisitedCell, VisitedNeighbour}),
        % Carve a door between the unvisited cell and visited neighbour
        Topology = get_topology(!.Maze),
        Edge = get_edge_between(Topology, UnvisitedCell, VisitedNeighbour),
        !:Maze = add_door(!.Maze, Edge),
        % Mark unvisited cell as visited
        set_tree234.insert(UnvisitedCell, !Visited),
        % Resume kill phase from the new cell
        kill_phase(UnvisitedCell, !Visited, !Maze, !RNG)
    ;
        MaybeTarget = no
        % No unvisited cells adjacent to visited cells - maze is complete
    ).

    % Find an unvisited cell that is adjacent to a visited cell.
    % Returns the unvisited cell and one of its visited neighbours.
    %
:- pred find_hunt_target(maze::in, set_tree234(cell)::in,
    maybe({cell, cell})::out) is det.

find_hunt_target(Maze, Visited, MaybeTarget) :-
    Cells = cells(Maze),
    find_hunt_target_loop(Maze, Visited, Cells, MaybeTarget).

:- pred find_hunt_target_loop(maze::in, set_tree234(cell)::in,
    list(cell)::in, maybe({cell, cell})::out) is det.

find_hunt_target_loop(_, _, [], no).
find_hunt_target_loop(Maze, Visited, [Cell | Rest], MaybeTarget) :-
    ( if set_tree234.member(Cell, Visited) then
        % Cell is already visited, skip it
        find_hunt_target_loop(Maze, Visited, Rest, MaybeTarget)
    else
        % Cell is unvisited - check if it has any visited neighbours
        VisitedNbrs = visited_neighbours(Maze, Visited, Cell),
        (
            VisitedNbrs = [Neighbour | _],
            % Found a target
            MaybeTarget = yes({Cell, Neighbour})
        ;
            VisitedNbrs = [],
            % No visited neighbours, keep searching
            find_hunt_target_loop(Maze, Visited, Rest, MaybeTarget)
        )
    ).

%---------------------------------------------------------------------------%
:- end_module generate.hunt_and_kill.
%---------------------------------------------------------------------------%
