%---------------------------------------------------------------------------%
% vim: ft=mercury ts=4 sw=4 et
%---------------------------------------------------------------------------%
% File: generate.dfs.m
%
% DFS (Recursive Backtracker) maze generation algorithm.
%
% 🤖 Generated with [Claude Code](https://claude.ai/code)
%
% Co-authored-by: Claude <noreply@anthropic.com>
%
%---------------------------------------------------------------------------%

:- module generate.dfs.
:- interface.

:- import_module maze.
:- import_module random.
:- import_module random.sfc16.

%---------------------------------------------------------------------------%

    % Generate a maze using the DFS (recursive backtracker) algorithm.
    %
:- pred generate_dfs(maze::in, maze::out,
    random.sfc16.random::in, random.sfc16.random::out) is det.

%---------------------------------------------------------------------------%
%---------------------------------------------------------------------------%

:- implementation.

:- import_module generate.
:- import_module grid.
:- import_module options.

:- import_module list.
:- import_module set_tree234.

%---------------------------------------------------------------------------%

generate_dfs(!Maze, !RNG) :-
    StartCell = get_start_cell(!.Maze),
    Visited0 = set_tree234.init,
    set_tree234.insert(StartCell, Visited0, Visited1),
    Stack0 = [],
    dfs_loop(StartCell, Stack0, Visited1, _, !Maze, !RNG).

    % Main DFS loop.
    % Current is the current cell.
    % Stack is the backtrack stack (cells we can return to).
    % Visited is the set of visited cells.
    %
:- pred dfs_loop(cell::in, list(cell)::in,
    set_tree234(cell)::in, set_tree234(cell)::out,
    maze::in, maze::out,
    random.sfc16.random::in, random.sfc16.random::out) is det.

dfs_loop(Current, Stack, !Visited, !Maze, !RNG) :-
    Unvisited = unvisited_neighbours(!.Maze, !.Visited, Current),
    (
        Unvisited = [_ | _],
        % Choose a random unvisited neighbour
        choose_random(Unvisited, Chosen, !RNG),
        % Carve a door between current and chosen
        Topology = get_topology(!.Maze),
        Edge = get_edge_between(Topology, Current, Chosen),
        !:Maze = add_door(!.Maze, Edge),
        % Mark chosen as visited
        set_tree234.insert(Chosen, !Visited),
        % Push current onto stack and move to chosen
        dfs_loop(Chosen, [Current | Stack], !Visited, !Maze, !RNG)
    ;
        Unvisited = [],
        % Dead end - backtrack
        (
            Stack = [Prev | Rest],
            dfs_loop(Prev, Rest, !Visited, !Maze, !RNG)
        ;
            Stack = []
            % Done - all cells visited
        )
    ).

%---------------------------------------------------------------------------%
:- end_module generate.dfs.
%---------------------------------------------------------------------------%
