%---------------------------------------------------------------------------%
% vim: ft=mercury ts=4 sw=4 et
%---------------------------------------------------------------------------%
% File: generate.binary_tree.m
%
% Binary Tree maze generation algorithm.
%
% 🤖 Generated with [Claude Code](https://claude.ai/code)
%
% Co-authored-by: Claude <noreply@anthropic.com>
%
%---------------------------------------------------------------------------%

:- module generate.binary_tree.
:- interface.

:- import_module maze.
:- import_module random.
:- import_module random.sfc16.

%---------------------------------------------------------------------------%

    % Generate a maze using the Binary Tree algorithm.
    %
    % For each cell, randomly carve a door either up or right.
    % At boundaries where only one direction is possible, use that direction.
    % This produces a maze with clear paths along the top and right edges.
    %
:- pred generate_binary_tree(maze::in, maze::out,
    random.sfc16.random::in, random.sfc16.random::out) is det.

%---------------------------------------------------------------------------%
%---------------------------------------------------------------------------%

:- implementation.

:- import_module grid.
:- import_module options.

:- import_module bool.
:- import_module int.
:- import_module list.

%---------------------------------------------------------------------------%

generate_binary_tree(!Maze, !RNG) :-
    Cells = cells(!.Maze),
    list.foldl2(process_cell, Cells, !Maze, !RNG).

    % Process a single cell: carve a door up or right if possible.
    %
:- pred process_cell(cell::in, maze::in, maze::out,
    random.sfc16.random::in, random.sfc16.random::out) is det.

process_cell(Cell, !Maze, !RNG) :-
    Topology = get_topology(!.Maze),
    % Check which directions are available (neighbour is in maze)
    UpNeighbour = adjacent(Topology, Cell, up),
    RightNeighbour = adjacent(Topology, Cell, right),
    ( if in_maze(!.Maze, UpNeighbour) then
        CanGoUp = yes
    else
        CanGoUp = no
    ),
    ( if in_maze(!.Maze, RightNeighbour) then
        CanGoRight = yes
    else
        CanGoRight = no
    ),
    (
        CanGoUp = yes,
        CanGoRight = yes,
        % Both directions available - choose randomly
        random.uniform_int_in_range(0, 2, Choice, !RNG),
        ( if Choice = 0 then
            Dir = up
        else
            Dir = right
        ),
        carve_door(Cell, Dir, !Maze)
    ;
        CanGoUp = yes,
        CanGoRight = no,
        % Only up available (right edge)
        carve_door(Cell, up, !Maze)
    ;
        CanGoUp = no,
        CanGoRight = yes,
        % Only right available (top edge)
        carve_door(Cell, right, !Maze)
    ;
        CanGoUp = no,
        CanGoRight = no
        % Top-right corner - no door to carve
    ).

    % Carve a door from a cell in the given direction.
    %
:- pred carve_door(cell::in, direction::in, maze::in, maze::out) is det.

carve_door(Cell, Dir, !Maze) :-
    Topology = get_topology(!.Maze),
    Edge = get_edge(Topology, Cell, Dir),
    !:Maze = add_door(!.Maze, Edge).

%---------------------------------------------------------------------------%
:- end_module generate.binary_tree.
%---------------------------------------------------------------------------%
