%---------------------------------------------------------------------------%
% vim: ft=mercury ts=4 sw=4 et
%---------------------------------------------------------------------------%
% File: generate.aldous_broder.m
%
% Aldous-Broder maze generation algorithm.
%
% 🤖 Generated with [Claude Code](https://claude.ai/code)
%
% Co-authored-by: Claude <noreply@anthropic.com>
%
%---------------------------------------------------------------------------%

:- module generate.aldous_broder.
:- interface.

:- import_module maze.
:- import_module random.
:- import_module random.sfc16.

%---------------------------------------------------------------------------%

    % Generate a maze using the Aldous-Broder algorithm.
    %
    % 1. Start from a random cell and mark it as visited.
    % 2. While there are unvisited cells:
    %    a. Move to a randomly chosen adjacent cell.
    %    b. If the new cell was not previously visited, carve a door from
    %       the previous cell and mark it as visited.
    %
    % This produces uniformly random spanning trees - every possible maze
    % is equally likely. However, the algorithm can be slow as it continues
    % random walking even over already-visited cells.
    %
:- pred generate_aldous_broder(maze::in, maze::out,
    random.sfc16.random::in, random.sfc16.random::out) is det.

%---------------------------------------------------------------------------%
%---------------------------------------------------------------------------%

:- implementation.

:- import_module generate.
:- import_module grid.
:- import_module options.

:- import_module int.
:- import_module list.
:- import_module set_tree234.

%---------------------------------------------------------------------------%

generate_aldous_broder(!Maze, !RNG) :-
    StartCell = get_start_cell(!.Maze),
    Visited0 = set_tree234.init,
    set_tree234.insert(StartCell, Visited0, Visited1),
    TotalCells = count_cells(!.Maze),
    random_walk(StartCell, 1, TotalCells, Visited1, _, !Maze, !RNG).

    % Count total cells in the maze.
    %
:- func count_cells(maze) = int.

count_cells(Maze) = list.length(cells(Maze)).

    % Random walk until all cells are visited.
    % Current is the current cell.
    % VisitedCount is the number of cells visited so far.
    % TotalCells is the total number of cells in the maze.
    %
:- pred random_walk(cell::in, int::in, int::in,
    set_tree234(cell)::in, set_tree234(cell)::out,
    maze::in, maze::out,
    random.sfc16.random::in, random.sfc16.random::out) is det.

random_walk(Current, VisitedCount, TotalCells, !Visited, !Maze, !RNG) :-
    ( if VisitedCount >= TotalCells then
        % All cells visited - done
        true
    else
        % Choose a random neighbour (must be in maze)
        Neighbours = valid_neighbours(!.Maze, Current),
        choose_random(Neighbours, Next, !RNG),
        % Check if next cell is unvisited
        ( if set_tree234.member(Next, !.Visited) then
            % Already visited - just move there, don't carve
            random_walk(Next, VisitedCount, TotalCells, !Visited, !Maze, !RNG)
        else
            % Unvisited - carve a door and mark as visited
            Topology = get_topology(!.Maze),
            Edge = get_edge_between(Topology, Current, Next),
            !:Maze = add_door(!.Maze, Edge),
            set_tree234.insert(Next, !Visited),
            random_walk(Next, VisitedCount + 1, TotalCells, !Visited, !Maze, !RNG)
        )
    ).

%---------------------------------------------------------------------------%
:- end_module generate.aldous_broder.
%---------------------------------------------------------------------------%
