%---------------------------------------------------------------------------%
% vim: ft=mercury ts=4 sw=4 et
%---------------------------------------------------------------------------%
% File: dump.m
%
% Dump maze representation to stdout.
%
% 🤖 Generated with [Claude Code](https://claude.ai/code)
%
% Co-authored-by: Claude <noreply@anthropic.com>
%
%---------------------------------------------------------------------------%

:- module dump.
:- interface.

:- import_module grid.
:- import_module io.
:- import_module list.
:- import_module maze.

%---------------------------------------------------------------------------%

    % Dump the maze to the given output stream.
    %
:- pred dump_maze(io.text_output_stream::in, maze::in, io::di, io::uo) is det.

    % Dump the solution path to the given output stream.
    %
:- pred dump_solution(io.text_output_stream::in, list(cell)::in,
    io::di, io::uo) is det.

%---------------------------------------------------------------------------%
%---------------------------------------------------------------------------%

:- implementation.

:- import_module int.
:- import_module options.
:- import_module string.

%---------------------------------------------------------------------------%

dump_maze(Stream, Maze, !IO) :-
    Topology = get_topology(Maze),
    Bounds = get_bounds(Maze),
    Width = Bounds ^ width,
    Height = Bounds ^ height,
    Entrance = get_entrance(Maze),
    Exit = get_exit(Maze),

    % Output basic details
    io.write_string(Stream, "Maze:\n", !IO),
    io.format(Stream, "  Topology: %s\n", [s(topology_to_string(Topology))], !IO),
    io.format(Stream, "  Size: %dx%d\n", [i(Width), i(Height)], !IO),
    io.format(Stream, "  Entrance: %s\n", [s(boundary_wall_to_string(Entrance))], !IO),
    io.format(Stream, "  Exit: %s\n", [s(boundary_wall_to_string(Exit))], !IO),

    % Output doors (internal edges with passages)
    Doors = list.filter(has_door(Maze), internal_edges(Maze)),
    io.format(Stream, "  Doors (%d):\n", [i(list.length(Doors))], !IO),
    list.foldl(dump_edge(Stream, Topology), Doors, !IO).

:- func topology_to_string(topology) = string.

topology_to_string(square) = "square".
topology_to_string(hex) = "hex".

:- func boundary_wall_to_string(boundary_wall) = string.

boundary_wall_to_string(boundary_wall(cell(X, Y), Dir)) = Str :-
    string.format("(%d,%d) %s", [i(X), i(Y), s(direction_to_string(Dir))], Str).

:- func direction_to_string(direction) = string.

direction_to_string(up) = "up".
direction_to_string(down) = "down".
direction_to_string(left) = "left".
direction_to_string(right) = "right".
direction_to_string(upper_left) = "upper_left".
direction_to_string(upper_right) = "upper_right".
direction_to_string(lower_left) = "lower_left".
direction_to_string(lower_right) = "lower_right".

:- func edge_to_string(topology, edge) = string.

edge_to_string(Topology, Edge) = Str :-
    edge_cells(Topology, Edge, Cell1, Cell2),
    Cell1 = cell(X1, Y1),
    Cell2 = cell(X2, Y2),
    Str = string.format("(%d,%d)-(%d,%d)", [i(X1), i(Y1), i(X2), i(Y2)]).

:- pred dump_edge(io.text_output_stream::in, topology::in, edge::in,
    io::di, io::uo) is det.

dump_edge(Stream, Topology, Edge, !IO) :-
    io.format(Stream, "    %s\n", [s(edge_to_string(Topology, Edge))], !IO).

%---------------------------------------------------------------------------%

dump_solution(Stream, Solution, !IO) :-
    io.format(Stream, "Solution (%d cells):\n", [i(list.length(Solution))], !IO),
    list.foldl(dump_cell(Stream), Solution, !IO).

:- pred dump_cell(io.text_output_stream::in, cell::in, io::di, io::uo) is det.

dump_cell(Stream, cell(X, Y), !IO) :-
    io.format(Stream, "  (%d,%d)\n", [i(X), i(Y)], !IO).

%---------------------------------------------------------------------------%
:- end_module dump.
%---------------------------------------------------------------------------%
