%---------------------------------------------------------------------------%
% vim: ft=mercury ts=4 sw=4 et
%---------------------------------------------------------------------------%
% File: check.m
%
% Maze validation and solution finding.
%
% 🤖 Generated with [Claude Code](https://claude.ai/code)
%
% Co-authored-by: Claude <noreply@anthropic.com>
%
%---------------------------------------------------------------------------%

:- module check.
:- interface.

:- import_module grid.
:- import_module list.
:- import_module maybe.
:- import_module maze.

%---------------------------------------------------------------------------%

    % Check that a maze is proper and return the solution path.
    %
    % A proper maze satisfies:
    % - Every cell is reachable from the start cell.
    % - The maze contains no loops.
    % - There is a path from the start cell to the target cell.
    %
    % Returns ok(Solution) if the maze is proper, or error(Message) if not.
    %
:- pred check_maze(maze::in, maybe_error(list(cell))::out) is det.

%---------------------------------------------------------------------------%
%---------------------------------------------------------------------------%

:- implementation.

:- import_module int.
:- import_module set_tree234.
:- import_module string.

%---------------------------------------------------------------------------%

check_maze(Maze, Result) :-
    StartCell = get_start_cell(Maze),
    TargetCell = get_target_cell(Maze),
    Visited0 = set_tree234.init,
    % Build path in reverse for O(1) prepend, reverse at end
    RevPathToHere = [StartCell],
    dfs_check(Maze, TargetCell, StartCell, RevPathToHere, Visited0, Visited,
        ok(no), DfsResult),

    (
        DfsResult = error(Msg),
        Result = error(Msg)
    ;
        DfsResult = ok(MaybeRevSolution),
        % Check that all cells were visited
        AllCells = cells(Maze),
        NumCells = list.length(AllCells),
        NumVisited = set_tree234.count(Visited),
        ( if NumVisited < NumCells then
            string.format("unreachable cells: visited %d of %d",
                [i(NumVisited), i(NumCells)], Msg),
            Result = error(Msg)
        else
            % Check that we found a solution
            (
                MaybeRevSolution = yes(RevSolution),
                Result = ok(list.reverse(RevSolution))
            ;
                MaybeRevSolution = no,
                Result = error("no solution found")
            )
        )
    ).

    % DFS traversal to check the maze and find the solution.
    %
    % RevPathToHere is the path from start to Current (inclusive), in reverse.
    % We build it in reverse for O(1) prepend, then reverse at the end.
    %
    % The result is ok(MaybeSolution) on success, or error(Msg) if a loop is
    % detected (which would indicate a bug in the maze generator).
    %
:- pred dfs_check(maze::in, cell::in, cell::in, list(cell)::in,
    set_tree234(cell)::in, set_tree234(cell)::out,
    maybe_error(maybe(list(cell)))::in,
    maybe_error(maybe(list(cell)))::out) is det.

dfs_check(Maze, TargetCell, Current, RevPathToHere, !Visited, !Result) :-
    (
        !.Result = error(_)
        % Already have an error, stop processing
    ;
        !.Result = ok(MaybeSolution0),
        Current = cell(X, Y),

        % Check for loop (cell already visited)
        ( if set_tree234.member(Current, !.Visited) then
            string.format("loop detected at cell (%d,%d)", [i(X), i(Y)], Msg),
            !:Result = error(Msg)
        else
            % Mark as visited
            set_tree234.insert(Current, !Visited),

            % Check if we reached the target
            ( if Current = TargetCell then
                !:Result = ok(yes(RevPathToHere))
            else
                !:Result = ok(MaybeSolution0)
            ),

            % Recurse to neighbours
            Neighbours = neighbours(Maze, Current),
            list.foldl2(
                visit_neighbour(Maze, TargetCell, RevPathToHere),
                Neighbours,
                !Visited,
                !Result)
        )
    ).

    % Visit a neighbour cell during DFS traversal if not already visited.
    %
:- pred visit_neighbour(maze::in, cell::in, list(cell)::in, cell::in,
    set_tree234(cell)::in, set_tree234(cell)::out,
    maybe_error(maybe(list(cell)))::in,
    maybe_error(maybe(list(cell)))::out) is det.

visit_neighbour(Maze, TargetCell, RevPathToHere, Neighbour, !Visited, !Result) :-
    (
        !.Result = error(_)
        % Already have an error, skip
    ;
        !.Result = ok(_),
        % Only visit if not already visited (avoid going back)
        ( if set_tree234.member(Neighbour, !.Visited) then
            true
        else
            dfs_check(Maze, TargetCell, Neighbour,
                [Neighbour | RevPathToHere], !Visited, !Result)
        )
    ).

%---------------------------------------------------------------------------%
:- end_module check.
%---------------------------------------------------------------------------%
