%---------------------------------------------------------------------------%
% vim: ft=mercury ts=4 sw=4 et
%---------------------------------------------------------------------------%
% File: boundary.m
%
% Boundary wall traversal for mazes.
%
% 🤖 Generated with [Claude Code](https://claude.ai/code)
%
% Co-authored-by: Claude <noreply@anthropic.com>
%
%---------------------------------------------------------------------------%

:- module boundary.
:- interface.

:- import_module list.
:- import_module maze.
:- import_module options.

%---------------------------------------------------------------------------%

    % boundary_walls(Topology, Bounds) = Walls
    %
    % Return the sequence of boundary walls tracing clockwise around the maze,
    % starting from the default entrance position (left edge, vertical center).
    %
:- func boundary_walls(topology, bounds) = list(boundary_wall).

    % split_boundary(Walls, Entrance, Exit, Segment1, Segment2)
    %
    % Split the boundary at the entrance and exit positions.
    % Segment1 runs clockwise from entrance to exit (excluding both).
    % Segment2 runs clockwise from exit to entrance (excluding both).
    %
:- pred split_boundary(list(boundary_wall)::in,
    boundary_wall::in, boundary_wall::in,
    list(boundary_wall)::out, list(boundary_wall)::out) is det.

    % wall_at_offset(Walls, Offset) = Wall
    %
    % Return the boundary wall at the given offset from the start of the list.
    % Offset 0 = first element (default entrance).
    % Positive offset = step clockwise.
    % Negative offset = step counter-clockwise.
    % Offset wraps around the boundary.
    %
:- func wall_at_offset(list(boundary_wall), int) = boundary_wall.

%---------------------------------------------------------------------------%
%---------------------------------------------------------------------------%

:- implementation.

:- import_module grid.
:- import_module int.

%---------------------------------------------------------------------------%

boundary_walls(Topology, Bounds) = Walls :-
    (
        Topology = square,
        Walls = square_boundary_walls(Bounds)
    ;
        Topology = hex,
        Walls = hex_boundary_walls(Bounds)
    ).

    % Generate boundary walls for a square grid, starting from the default
    % entrance position and tracing clockwise.
    %
:- func square_boundary_walls(bounds) = list(boundary_wall).

square_boundary_walls(Bounds) = Walls :-
    Width = Bounds ^ width,
    Height = Bounds ^ height,

    % Default entrance is at (0, Height/2) pointing left
    StartY = Height / 2,

    % Trace clockwise from default entrance:
    % 1. Start at entrance, go up the left edge (Y decreasing, direction left)
    % 2. Across the top edge (X increasing, direction up)
    % 3. Down the right edge (Y increasing, direction right)
    % 4. Across the bottom edge (X decreasing, direction down)
    % 5. Up the left edge back to entrance (Y decreasing, direction left)

    % Left edge: from StartY up to 0 (includes entrance)
    LeftUpper = list.map(
        (func(Y) = boundary_wall(cell(0, Y), left)),
        reverse(0 .. StartY)),

    % Top edge: from 0 to Width-1
    Top = list.map(
        (func(X) = boundary_wall(cell(X, 0), up)),
        0 .. (Width - 1)),

    % Right edge: from 0 to Height-1
    Right = list.map(
        (func(Y) = boundary_wall(cell(Width - 1, Y), right)),
        0 .. (Height - 1)),

    % Bottom edge: from Width-1 down to 0
    Bottom = list.map(
        (func(X) = boundary_wall(cell(X, Height - 1), down)),
        reverse(0 .. (Width - 1))),

    % Left edge: from Height-1 up to StartY+1 (excludes entrance)
    LeftLower = list.map(
        (func(Y) = boundary_wall(cell(0, Y), left)),
        reverse((StartY + 1) .. (Height - 1))),

    Walls = LeftUpper ++ Top ++ Right ++ Bottom ++ LeftLower.

    % Generate boundary walls for a hex grid, starting from the default
    % entrance position and tracing clockwise.
    %
    % The hex maze forms a pointy-topped hexagon with corners at clock positions:
    %   10 o'clock: (0, 0)
    %   12 o'clock: ((W-1)/2, 0)
    %   2 o'clock:  (W-1, (W-1)/2)
    %   4 o'clock:  (W-1, (W-1)/2 + H - 1)
    %   6 o'clock:  ((W-1)/2, (W-1)/2 + H - 1)
    %   8 o'clock:  (0, H - 1)
    %
    % Default entrance is at (0, H/2) on the left edge (8-10 o'clock side).
    %
    % Each boundary cell contributes 2 edges (or 3 for corner cells) to the
    % boundary. We emit one boundary_wall per edge, not per cell, so that
    % the boundary traces the zigzag path around the hex maze.
    %
:- func hex_boundary_walls(bounds) = list(boundary_wall).

hex_boundary_walls(Bounds) = Walls :-
    W = Bounds ^ width,
    H = Bounds ^ height,
    HalfW = (W - 1) / 2,

    % Default entrance is at (0, H/2), first edge is lower_left
    StartR = H / 2,

    % Trace clockwise from default entrance.
    % Each cell on a straight edge contributes 2 boundary edges.
    % Corner cells contribute 3 boundary edges.

    % Segment 1: Left edge from entrance up to 10 o'clock corner
    % Cells (0, R) for R from StartR down to 0
    % Each cell contributes: lower_left, upper_left
    % (except we start mid-edge at the entrance)
    Seg1 = list.condense(list.map(
        (func(R) = [boundary_wall(cell(0, R), lower_left),
                    boundary_wall(cell(0, R), upper_left)]),
        reverse(0 .. StartR))),

    % Segment 2: 10 o'clock corner (0, 0) contributes up edge
    % (lower_left and upper_left already emitted in Seg1)
    Seg2 = [boundary_wall(cell(0, 0), up)],

    % Segment 3: Top-left edge from 10 to 12 o'clock
    % Cells (Q, 0) for Q from 1 to HalfW
    % Each cell contributes: upper_left, up
    Seg3 = list.condense(list.map(
        (func(Q) = [boundary_wall(cell(Q, 0), upper_left),
                    boundary_wall(cell(Q, 0), up)]),
        1 .. HalfW)),

    % Segment 4: 12 o'clock corner (HalfW, 0) contributes upper_right edge
    % (upper_left and up already emitted in Seg3)
    Seg4 = [boundary_wall(cell(HalfW, 0), upper_right)],

    % Segment 5: Top-right edge from 12 to 2 o'clock
    % Cells (Q, Q - HalfW) for Q from HalfW+1 to W-1
    % Each cell contributes: up, upper_right
    Seg5 = list.condense(list.map(
        (func(Q) = [boundary_wall(cell(Q, Q - HalfW), up),
                    boundary_wall(cell(Q, Q - HalfW), upper_right)]),
        (HalfW + 1) .. (W - 1))),

    % Segment 6: 2 o'clock corner (W-1, HalfW) contributes lower_right edge
    % (up and upper_right already emitted in Seg5)
    Seg6 = [boundary_wall(cell(W - 1, HalfW), lower_right)],

    % Segment 7: Right edge from 2 to 4 o'clock
    % Cells (W-1, R) for R from HalfW+1 to HalfW+H-1
    % Each cell contributes: upper_right, lower_right
    Seg7 = list.condense(list.map(
        (func(R) = [boundary_wall(cell(W - 1, R), upper_right),
                    boundary_wall(cell(W - 1, R), lower_right)]),
        (HalfW + 1) .. (HalfW + H - 1))),

    % Segment 8: 4 o'clock corner (W-1, HalfW+H-1) contributes down edge
    % (upper_right and lower_right already emitted in Seg7)
    Seg8 = [boundary_wall(cell(W - 1, HalfW + H - 1), down)],

    % Segment 9: Bottom-right edge from 4 to 6 o'clock
    % Cells (Q, HalfW + H - 1) for Q from W-2 down to HalfW
    % Each cell contributes: lower_right, down
    Seg9 = list.condense(list.map(
        (func(Q) = [boundary_wall(cell(Q, HalfW + H - 1), lower_right),
                    boundary_wall(cell(Q, HalfW + H - 1), down)]),
        reverse(HalfW .. (W - 2)))),

    % Segment 10: 6 o'clock corner (HalfW, HalfW+H-1) contributes lower_left edge
    % (lower_right and down already emitted in Seg9)
    Seg10 = [boundary_wall(cell(HalfW, HalfW + H - 1), lower_left)],

    % Segment 11: Bottom-left edge from 6 to 8 o'clock
    % Cells (Q, Q + H - 1) for Q from HalfW-1 down to 0
    % Each cell contributes: down, lower_left
    Seg11 = list.condense(list.map(
        (func(Q) = [boundary_wall(cell(Q, Q + H - 1), down),
                    boundary_wall(cell(Q, Q + H - 1), lower_left)]),
        reverse(0 .. (HalfW - 1)))),

    % Segment 12: 8 o'clock corner (0, H-1) contributes upper_left edge
    % (down and lower_left already emitted in Seg11)
    Seg12 = [boundary_wall(cell(0, H - 1), upper_left)],

    % Segment 13: Left edge from 8 o'clock back to entrance
    % Cells (0, R) for R from H-2 down to StartR+1
    % Each cell contributes: lower_left, upper_left
    Seg13 = list.condense(list.map(
        (func(R) = [boundary_wall(cell(0, R), lower_left),
                    boundary_wall(cell(0, R), upper_left)]),
        reverse((StartR + 1) .. (H - 2)))),

    Walls = Seg1 ++ Seg2 ++ Seg3 ++ Seg4 ++ Seg5 ++ Seg6 ++ Seg7 ++ Seg8 ++
            Seg9 ++ Seg10 ++ Seg11 ++ Seg12 ++ Seg13.

%---------------------------------------------------------------------------%

split_boundary(Walls, Entrance, Exit, Segment1, Segment2) :-
    % Rotate so Entrance is at the front
    rotate_to(Entrance, Walls, RotatedWalls),
    % Remove the entrance, then take until we hit exit
    (
        RotatedWalls = [_ | AfterEntrance],
        take_until(Exit, AfterEntrance, Segment1, AtExit),
        % Remove the exit, rest is Segment2
        (
            AtExit = [_ | Segment2]
        ;
            AtExit = [],
            Segment2 = []
        )
    ;
        RotatedWalls = [],
        Segment1 = [],
        Segment2 = []
    ).

    % Rotate the list so the given element is at the front.
    %
:- pred rotate_to(boundary_wall::in, list(boundary_wall)::in,
    list(boundary_wall)::out) is det.

rotate_to(_, [], []).
rotate_to(Target, [H | T], Result) :-
    ( if H = Target then
        Result = [H | T]
    else
        rotate_to(Target, T ++ [H], Result)
    ).

    % Take elements until we hit the target (target not included in first list).
    %
:- pred take_until(boundary_wall::in, list(boundary_wall)::in,
    list(boundary_wall)::out, list(boundary_wall)::out) is det.

take_until(_, [], [], []).
take_until(Target, [H | T], Before, AtAndAfter) :-
    ( if H = Target then
        Before = [],
        AtAndAfter = [H | T]
    else
        take_until(Target, T, BeforeRest, AtAndAfter),
        Before = [H | BeforeRest]
    ).

%---------------------------------------------------------------------------%

wall_at_offset(Walls, Offset) = Wall :-
    Len = list.length(Walls),
    ( if Len = 0 then
        % Should not happen for valid mazes
        Wall = boundary_wall(cell(0, 0), left)
    else
        % Normalize offset to [0, Len)
        NormOffset = ((Offset mod Len) + Len) mod Len,
        list.det_index0(Walls, NormOffset, Wall)
    ).

%---------------------------------------------------------------------------%
:- end_module boundary.
%---------------------------------------------------------------------------%
