%---------------------------------------------------------------------------%
% vim: ft=mercury ts=4 sw=4 et
%---------------------------------------------------------------------------%
% File: amaze.m
%
% Main module for the maze generator.
%
% 🤖 Generated with [Claude Code](https://claude.ai/code)
%
% Co-authored-by: Claude <noreply@anthropic.com>
%
%---------------------------------------------------------------------------%

:- module amaze.
:- interface.

:- import_module io.

:- pred main(io::di, io::uo) is det.

%---------------------------------------------------------------------------%
%---------------------------------------------------------------------------%

:- implementation.

:- import_module boundary.
:- import_module check.
:- import_module dump.
:- import_module generate.
:- import_module grid.
:- import_module maze.
:- import_module options.
:- import_module render.

:- import_module bool.
:- import_module int.
:- import_module list.
:- import_module maybe.
:- import_module random.
:- import_module random.sfc16.
:- import_module string.
:- import_module uint64.

%---------------------------------------------------------------------------%

main(!IO) :-
    io.command_line_arguments(Args, !IO),
    parse_options(Args, MaybeOptions, !IO),
    (
        MaybeOptions = yes(Options),
        run(Options, !IO)
    ;
        MaybeOptions = no,
        % Error message already printed by parse_options
        io.set_exit_status(1, !IO)
    ).

:- pred run(options::in, io::di, io::uo) is det.

run(Options, !IO) :-
    Topology = Options ^ opt_topology,
    Width = Options ^ opt_width,
    Height = Options ^ opt_height,
    EntranceOffset = Options ^ opt_entrance_offset,
    ExitOffset = Options ^ opt_exit_offset,

    Bounds = bounds(Width, Height),

    % Get the boundary walls sequence and calculate entrance/exit from offsets
    Walls = boundary_walls(Topology, Bounds),
    BoundaryLen = list.length(Walls),

    % Entrance is at the given offset from the default (centre of left edge)
    Entrance = wall_at_offset(Walls, EntranceOffset),

    % Exit is at the given offset from the default exit position
    % Default exit is halfway around the boundary from the default entrance
    DefaultExitOffset = BoundaryLen / 2,
    Exit = wall_at_offset(Walls, DefaultExitOffset + ExitOffset),

    % Create empty maze
    Maze0 = maze.init(Topology, Bounds, Entrance, Exit),

    % Seed the RNG from the command-line seed
    Seed = Options ^ opt_random_seed,
    RNG0 = sfc16.seed(uint64.cast_from_int(Seed)),

    % Generate the maze
    Algorithm = Options ^ opt_algorithm,
    generate(Algorithm, Maze0, Maze, RNG0, _RNG),

    % Check the maze and get the solution
    check_maze(Maze, CheckResult),
    (
        CheckResult = error(Msg),
        io.stderr_stream(Stderr, !IO),
        io.format(Stderr, "error: %s\n", [s(Msg)], !IO),
        io.set_exit_status(1, !IO)
    ;
        CheckResult = ok(Solution),
        output_maze(Options, Maze, Solution, !IO)
    ).

:- pred output_maze(options::in, maze::in, list(cell)::in,
    io::di, io::uo) is det.

output_maze(Options, Maze, Solution, !IO) :-
    Dump = Options ^ opt_dump,
    (
        Dump = yes,
        % Dump the maze and solution to stdout
        io.stdout_stream(Stdout, !IO),
        dump_maze(Stdout, Maze, !IO),
        dump_solution(Stdout, Solution, !IO)
    ;
        Dump = no,
        % Construct render options from command-line options
        RenderOpts = render_options(
            Options ^ opt_cell_spacing,
            Options ^ opt_margin,
            Options ^ opt_maze_bg_color,
            Options ^ opt_cell_bg_color,
            Options ^ opt_internal_wall_color,
            Options ^ opt_boundary_wall_color,
            Options ^ opt_path_color,
            Options ^ opt_internal_wall_width,
            Options ^ opt_boundary_wall_width,
            Options ^ opt_path_width
        ),
        % Only include solution if --path option given
        ShowPath = Options ^ opt_show_path,
        (
            ShowPath = yes,
            PathToRender = Solution
        ;
            ShowPath = no,
            PathToRender = []
        ),
        % Render the maze to output file
        (
            Options ^ opt_output_file = yes(OutputFile)
        ;
            Options ^ opt_output_file = no,
            OutputFile = "output.svg"
        ),
        render_maze(OutputFile, RenderOpts, Maze, PathToRender, !IO)
    ).

%---------------------------------------------------------------------------%
:- end_module amaze.
%---------------------------------------------------------------------------%
