# Implementation Plan

## System Requirements

This will be a standalone project written in Mercury.

### Mercury Documentation

- The [Mercury Language Reference Manual](https://mercurylang.org/information/doc-latest/mercury_reference_manual/) contains the full language specification.

Important sections of the reference manual:

- [Syntax](https://mercurylang.org/information/doc-latest/mercury_reference_manual/Syntax.html)
- [Clauses](https://mercurylang.org/information/doc-latest/mercury_reference_manual/Clauses.html)
- [Types](https://mercurylang.org/information/doc-latest/mercury_reference_manual/Types.html)
- [Modes](https://mercurylang.org/information/doc-latest/mercury_reference_manual/Modes.html)
- [Destructive update](https://mercurylang.org/information/doc-latest/mercury_reference_manual/Destructive-update.html)
- [Determinism](https://mercurylang.org/information/doc-latest/mercury_reference_manual/Determinism.html)
- [Higher-order programming](https://mercurylang.org/information/doc-latest/mercury_reference_manual/Higher-order-programming.html)
- [Modules](https://mercurylang.org/information/doc-latest/mercury_reference_manual/Modules.html)
- [Typeclasses](https://mercurylang.org/information/doc-latest/mercury_reference_manual/Typeclasses.html)
- [Exception handling](https://mercurylang.org/information/doc-latest/mercury_reference_manual/Exception-handling.html)

The locally installed Mercury Standard Library can be found in `~/mlib`. Refer to this for library interfaces and documentation.

### Mercury Usage

Compile an individual module:

```
mmc --make foo.c
```

Compile the main program (and all dependencies):

```
mmc --make amaze
```

Clean up build artifacts:

```
mmc --make amaze.realclean
```

### Notes

Some standard library predicates have both a function version and a predicate version with the same name. For example, `string.format` has:

```mercury
:- func format(string, list(poly_type)) = string.
:- pred format(string::in, list(poly_type)::in, string::out) is det.
```

When calling such a predicate with fewer arguments than the predicate version expects, Mercury may interpret the call as a partial application (a higher-order term) rather than a function call. For example, `string.format("x = %d", [i(X)])` could be interpreted as either:

- A function call returning a `string`
- A partial application of the predicate, with type `pred(string)`

To avoid this ambiguity, use the predicate version with all arguments:

```mercury
string.format("x = %d", [i(X)], Result)
```

## Core Concepts

### Grid Topology

1. A grid is either square or hexagonal. The grid is made of cells (squares or hexagons) that are separated from each other by edges. The grid conceptually extends to infinity.

2. Each grid topology is described by a set of directions, either 4 or 6, which map each cell to one of its adjacent cells.

3. An edge can be identified by a cell and a direction. The same edge can be approached from opposite directions, which means there are two ways of identifying each edge.

4. Directions come in opposite pairs (e.g., left/right, up/down). By designating one direction from each pair as "canonical," every edge can be uniquely identified by a cell and a canonical direction. This is the edge between that cell and its neighbour in the canonical direction.

### Mazes

1. The boundary of a maze is a sequence of edges forming a loop. Cells enclosed by the loop are part of the maze.

2. For each cell in the maze, for each direction, there are two possibilities:

   - the adjacent cell in that direction is also part of the maze, in which case the edge separating them is an internal wall
   - the adjacent cell in that direction is not part of the maze, in which case the edge separating them is a boundary wall

3. Each internal wall may or may not have a door. If there is a door then it allows passage between cells, from either direction.

4. Each boundary wall may or may not have a door. Each maze has two such doors: an entrance and an exit. The cell attached to the entrance is the start cell, and the cell attached to the exit is the target cell.

5. A path is a sequence of cells with no repeats, except that the last may be the same as the first. Each consecutive pair in the sequence is adjacent and the wall between them has a door.

6. A cell A is reachable from a cell B if there exists a path from B to A.

7. A path is a loop if it has three or more distinct cells, and the last cell is the same as the first.

8. A solution to the maze is a path from the start cell to the target cell.

9. A proper maze satisfies the following:

   - every cell in the maze is reachable from the start cell
   - the maze contains no loops
   - the entrance is different to the exit

## Functional Requirements

### Overview

The program takes a set of input and output parameters, randomly generates a maze according to the input parameters, then produces an output file containing the maze.

#### Output Formats

The output file can be either:

- An SVG file depicting the maze in accordance with the output parameters.
- A text dump of data representing the doors of the maze.

#### Input Parameters

- grid topology (square or hex)
- grid size/dimensions
- entrance and exit locations
- random seed (default to be derived from system time)
- maze generation algorithm

#### Output Parameters

- output file name
- cell spacing (the distance in viewbox units between the center of adjacent cells)
- size of margin around the maze
- whether or not to display the solution path
- colors: maze background, cell background, internal wall, boundary wall, solution path
- stroke widths: internal wall, boundary wall, solution path

### Command Line Interface

```
amaze <options> [--] output.svg
```

#### Input Options

- `-t, --topology <square|hex>` - Grid topology (default: square)
- `-s, --size <N|WxH>` - Maze size (default: 10)
  - For square grids: `N` means `NxN`
  - For hex grids: `N` means `W = 2*N - 1`, `H = N` (regular hexagon shape)
- `-r, --random-seed <number>` - Random number seed (default: derived from system time)
- `-e, --entrance-offset <number>` - Entrance offset from centre (default: 0)
- `-x, --exit-offset <number>` - Exit offset from centre (default: same as entrance offset)
- `-a, --algorithm <name>` - Maze generation algorithm (default: kruskal)

#### Output Options

- `--cell-spacing <number>` - Cell spacing in viewbox units (default: 10)
- `--margin <number>` - Margin in multiples of cell spacing (default: 2)
- `-p, --path` - Display the solution path
- `--maze-bg-color <color>` - Maze background color (default: transparent)
- `--cell-bg-color <color>` - Cell background color (default: white)
- `--internal-wall-color <color>` - Internal wall color (default: black)
- `--boundary-wall-color <color>` - Boundary wall color (default: black)
- `--path-color <color>` - Solution path color (default: red)
- `--internal-wall-width <number>` - Internal wall stroke width (default: 1)
- `--boundary-wall-width <number>` - Boundary wall stroke width (default: 2)
- `--path-width <number>` - Solution path stroke width (default: 3)

Color values are passed through to the SVG output without validation. Any valid SVG color string is accepted (e.g., `red`, `#ff0000`, `rgb(255,0,0)`).

#### General Options

- `-h, --help` - Display usage
- `-d, --dump` - Dump the maze representation to stdout

### SVG Rendering

- Lines in the output SVG must have rounded ends and corners.

Rendering order (bottom layer to top):

1. **Maze background.** A rectangle covering the entire viewbox.
2. **Cell background.** The boundary polygon filled with the cell background color.
3. **Internal walls.** Each wall is rendered as a line along the relevant edge (between two corners of the relevant cell).
4. **Boundary walls.** Rendered everywhere except the two openings (for example using two polylines).
5. **Solution path** (if present). A polyline joining the centre of each cell in the solution path.

### Maze Generation Algorithms

#### Recursive Backtracker (Depth-First Search)

1. Start from a random cell and mark it as visited.
2. While there are unvisited cells:
   a. If the current cell has unvisited neighbours, randomly choose one, carve a door between them, move to it, and mark it as visited.
   b. Otherwise, backtrack to the previous cell.

Characteristics: Produces mazes with long, winding corridors and relatively few dead ends. The algorithm creates a strong bias toward longer paths. Simple to implement using a stack or recursion.

#### Kruskal's Algorithm

1. Create a list of all potential internal walls (edges between adjacent cells).
2. Shuffle the list randomly.
3. For each wall in the list, if the cells on either side are not yet connected (belong to different sets), remove the wall (add a door) and merge their sets.

Characteristics: Produces more uniform mazes with less bias than recursive backtracker. Requires a union-find (disjoint set) data structure for efficient implementation. The maze grows from many points simultaneously rather than from a single path.

#### Prim's Algorithm

1. Start with a grid where all walls are present.
2. Pick a random starting cell and add it to the maze.
3. Add all walls of that cell to a wall list.
4. While there are walls in the list:
   a. Pick a random wall from the list.
   b. If the cell on the opposite side isn't in the maze yet, remove the wall (add a door) and add that cell to the maze.
   c. Add the new cell's walls to the wall list.
   d. Remove the wall from the list.

Characteristics: Produces mazes similar to Kruskal's but with a different growth pattern—the maze expands outward from a central region. Tends to have shorter dead ends and more branching near the starting point.

#### Aldous-Broder Algorithm

1. Start from a random cell and mark it as visited.
2. While there are unvisited cells:
   a. Move to a randomly chosen adjacent cell.
   b. If the new cell was not previously visited, carve a door from the previous cell and mark it as visited.

Characteristics: Produces uniformly random spanning trees—every possible maze is equally likely. However, the algorithm is inefficient because it continues random walking even over already-visited cells. Completion time is unpredictable and can be slow for large mazes.

#### Wilson's Algorithm

1. Mark a random cell as part of the maze.
2. While there are cells not in the maze:
   a. Start a random walk from a random unvisited cell.
   b. Walk randomly until hitting a cell that's already in the maze, erasing any loops formed during the walk.
   c. Add the loop-erased path to the maze, carving doors along the path.

Characteristics: Also produces uniformly random spanning trees like Aldous-Broder, but more efficiently. The loop-erasing prevents wasted work. Initial walks may be slow, but the algorithm speeds up as more cells join the maze.

#### Hunt-and-Kill Algorithm

1. Start from a random cell and mark it as visited.
2. Perform a random walk, carving doors to unvisited neighbours.
3. When stuck (no unvisited neighbours):
   a. Scan the grid for an unvisited cell adjacent to a visited cell.
   b. If found, carve a door between them and continue the walk from there.
   c. If not found, the maze is complete.

Characteristics: Produces results similar to recursive backtracker but without requiring a stack for backtracking. The "hunt" phase (scanning for a new starting point) can be slow, but the algorithm uses less memory. Tends to produce mazes with a more uniform texture than recursive backtracker.

#### Binary Tree Algorithm

1. For each cell in the grid:
   a. Randomly choose to carve a door either north or east (or the two available directions at boundaries).
   b. If only one direction is possible (corner/edge cells), use that direction.

Characteristics: Extremely fast and simple—O(n) time with no additional data structures. However, produces heavily biased mazes: there's always a clear path along the north and east edges, and the maze has a strong diagonal texture. Good for learning but produces lower-quality mazes.

**Hex compatibility:** This algorithm is square-only. If selected with hex topology, the program should report an error.

#### Sidewinder Algorithm

1. For each row, from bottom to top:
   a. For each cell in the row, decide randomly whether to carve east or close out the current "run."
   b. When closing a run, randomly choose one cell from the run and carve north from it.
   c. At the eastern boundary, always close the run.
2. The top row is handled specially: always carve east (no north option).

Characteristics: Similar speed to binary tree but with different bias. Produces mazes with a clear path along the top edge and vertical bias elsewhere. More interesting than binary tree but still noticeably biased. Works row-by-row, making it memory-efficient.

**Hex compatibility:** This algorithm is square-only. If selected with hex topology, the program should report an error.

## Architecture

### Module Structure

```
amaze (main module)
├── options (command-line parsing and program parameters)
├── grid (grid topology: cells, directions, adjacency, edges)
├── maze (maze data structure and operations)
├── boundary (boundary wall traversal)
├── generate (maze generation)
│   ├── generate.dfs (DFS / recursive backtracker)
│   ├── generate.kruskal (Kruskal's algorithm)
│   ├── generate.prim (Prim's algorithm)
│   └── ... (other algorithms)
├── check (validation and solution finding)
├── render (SVG output)
└── dump (pretty-print maze data representation)
```

### Module Responsibilities

- **amaze** - Main entry point; orchestrates option parsing, maze generation, and output.

- **options** - Command-line parsing and option handling. Imported by any module that depends on program parameters.

- **grid** - Encapsulates grid topology (cells, directions, adjacency, edges) for both square and hexagonal grids. This maps directly to the "Grid Topology" core concepts.

- **maze** - Defines the maze structure (cells in maze, walls, doors, entrance/exit). Uses the grid module but adds maze-specific concepts like walls and doors.

- **boundary** - Generates the sequence of boundary edges tracing clockwise around the maze. Used for calculating entrance/exit positions from offsets and for rendering boundary walls.

- **generate** - Each algorithm gets its own submodule, sharing a common interface. This makes it easy to add new algorithms and select one at runtime.

- **check** - Validates maze properties (all cells reachable, no loops) and finds the solution path. Combines validation and solving in a single depth-first traversal.

- **render** - SVG output generation.

- **dump** - Pretty-prints the maze data representation to stdout.

### Design Rationale

- Clean separation between grid topology and maze concepts.
- Algorithms are pluggable via a common interface.
- Grid module handles both square and hex topologies, making the rest of the code topology-agnostic.
- Rendering is decoupled from maze structure.
- Validation is separate, aiding development and debugging.

## Detailed Design

### Grids

The grid module provides a purely geometric abstraction over square and hexagonal grids. It knows nothing about mazes, walls, or doors—it simply answers questions about cells, directions, adjacency, and edges.

The grid module imports the `topology` type from the `options` module:

```mercury
:- type topology ---> square ; hex.
```

#### Types

**Cell:** A position on the grid, represented as integer coordinates.

```mercury
:- type cell ---> cell(int, int).
```

For square grids this is `(X, Y)` where:
- X increases in the `right` direction (basis vector for X)
- Y increases in the `down` direction (basis vector for Y)

For hex grids (flat-top cells) this uses axial coordinates `(Q, R)` where:
- Q increases in the `upper_right` direction (basis vector for Q)
- R increases in the `down` direction (basis vector for R)

The choice of basis vectors determines how coordinates map to directions. By definition, the basis direction for a coordinate axis corresponds to a delta of `(1, 0)` or `(0, 1)` respectively. Other directions are linear combinations of the basis vectors.

**Direction:** The directions connecting adjacent cells.

```mercury
:- type direction
    --->    up
    ;       down
    ;       left          % square only
    ;       right         % square only
    ;       upper_left    % hex only
    ;       upper_right   % hex only
    ;       lower_left    % hex only
    ;       lower_right.  % hex only
```

- Square grids use: `up`, `down`, `left`, `right`
- Hex grids (flat-top) use: `up`, `down`, `upper_left`, `upper_right`, `lower_left`, `lower_right`

Opposite pairs:

- `up` ↔ `down`
- `left` ↔ `right`
- `upper_left` ↔ `lower_right`
- `upper_right` ↔ `lower_left`

#### Hex Adjacency

For a hex cell `(Q, R)`, the adjacent cells are:

| Direction     | Adjacent Cell   |
|---------------|-----------------|
| `up`          | `(Q, R - 1)`    |
| `down`        | `(Q, R + 1)`    |
| `upper_right` | `(Q + 1, R)`    |
| `lower_left`  | `(Q - 1, R)`    |
| `upper_left`  | `(Q - 1, R - 1)`|
| `lower_right` | `(Q + 1, R + 1)`|

This table is derived from the basis vectors: `upper_right` → `(+1, 0)` and `down` → `(0, +1)`. The other directions are linear combinations of the basis vectors and their opposites:
- `lower_right` = `upper_right` + `down` = `(+1, +1)`
- `upper_left` = `lower_left` + `up` = `(-1, -1)`

**Edge:** An edge between two adjacent cells. The representation is an implementation detail; other modules construct edges via provided functions.

```mercury
:- type edge.
```

#### Operations

- `directions(topology) = list(direction)` - List all directions for the topology (4 or 6).
- `canonical_directions(topology) = list(direction)` - List the canonical directions (2 or 3).
  - Square: `up`, `right`
  - Hex: `up`, `upper_right`, `lower_right`
- `opposite(direction) = direction` - Return the opposite direction.
- `adjacent(topology, cell, direction) = cell` - Return the adjacent cell in the given direction. This is the authoritative definition of the coordinate system; all other code that needs direction-to-coordinate mappings should derive them from this function.
- `direction_delta(topology, direction, delta_x, delta_y)` - Return the coordinate delta for a direction. Derived from `adjacent` to ensure consistency.
- `get_edge_between(topology, cell, cell) = edge` - Return the edge between two adjacent cells. Throws an exception if the cells are not adjacent.
- `get_edge(topology, cell, direction) = edge` - Return the edge between a cell and its neighbour in the given direction.

### Mazes

The maze module builds on the grid module to represent a finite maze with walls, doors, an entrance, and an exit.

#### Types

**Bounds:** The dimensions of a maze.

```mercury
:- type bounds ---> bounds(width :: int, height :: int).
```

The interpretation of width and height depends on the topology.

**Square bounds:** A cell `cell(X, Y)` is in the maze if:
- `0 <= X < W`
- `0 <= Y < H`

This forms a W × H rectangle.

**Hex bounds:** The maze forms a pointy-topped hexagon shape (rotated 90° from the flat-topped cells). The dimensions are interpreted as:
- W = width: the number of cells in a horizontal path from the left edge to the right edge
- H = height: the number of cells stacked vertically on the left and right edges

A cell `cell(Q, R)` is in the maze if all of the following hold:
- `0 <= Q < W`
- `Q - (W-1)/2 <= R` (top-right diagonal constraint)
- `R <= Q + H - 1` (bottom-left diagonal constraint)

Note: The second coordinate R ranges over `[0, (W-1)/2 + H)` but is further constrained by the diagonal edges.

The six corners of the hex maze (using clock positions) are:
- 10 o'clock: `(0, 0)`
- 12 o'clock: `((W-1)/2, 0)`
- 2 o'clock: `(W-1, (W-1)/2)`
- 4 o'clock: `(W-1, (W-1)/2 + H - 1)`
- 6 o'clock: `((W-1)/2, (W-1)/2 + H - 1)`
- 8 o'clock: `(0, H - 1)`

**Maze:** The complete maze structure.

```mercury
:- type maze
    --->    maze(
                topology :: topology,
                bounds :: bounds,
                doors :: set_tree234(edge),
                start_cell :: cell,
                start_direction :: direction,
                target_cell :: cell,
                target_direction :: direction
            ).
```

- `topology` - The grid topology (square or hex), imported from options.
- `bounds` - The maze dimensions.
- `doors` - The set of internal edges that have doors (passages between cells). Uses `set_tree234` for O(log n) insert and lookup operations.
- `start_cell` - The cell at the entrance of the maze.
- `start_direction` - The direction from the start cell towards the outside of the maze.
- `target_cell` - The cell at the exit of the maze.
- `target_direction` - The direction from the target cell towards the outside of the maze.

The entrance and exit edges can be derived using `get_edge(topology, start_cell, start_direction)` and `get_edge(topology, target_cell, target_direction)` respectively.

Walls are implicit: any internal edge not in the `doors` set is a wall.

#### Operations

- `init(topology, bounds, start_cell, start_direction, target_cell, target_direction) = maze` - Create a maze with no internal doors (all walls).
- `in_maze(maze, cell)` - Is this cell within the maze bounds?
- `has_door(maze, edge)` - Does this edge have a door?
- `add_door(maze, edge) = maze` - Add a door to an internal edge.
- `cells(maze) = list(cell)` - List all cells in the maze.
- `internal_edges(maze) = list(edge)` - List all internal edges.
- `entrance(maze) = edge` - The entrance edge (derived from start cell and direction).
- `exit(maze) = edge` - The exit edge (derived from target cell and direction).
- `neighbours(maze, cell) = list(cell)` - Adjacent cells reachable through doors.

### Boundary

The boundary module generates the sequence of boundary edges tracing clockwise around the maze. This sequence is used for:

1. Calculating entrance/exit cell positions from command-line offsets
2. Rendering boundary walls as polylines (split at entrance/exit openings)

#### Boundary Edge Sequence

The boundary consists of all edges between cells inside the maze and cells outside the maze. We trace clockwise starting from the default entrance position.

##### Square Grid Boundary

**Default entrance:** The cell at the vertical centre of the left edge, pointing left. For a maze of height H, this is cell `(0, H/2)` with direction `left`.

**Default exit:** Halfway around the boundary from the entrance. For a rectangular maze, this places it on the right edge at cell `(W-1, (H-1)/2)` with direction `right`.

**Traversal order:** Starting from the default entrance, trace clockwise:
1. Up the left edge (direction `left` for each cell)
2. Across the top edge (direction `up` for each cell)
3. Down the right edge (direction `right` for each cell)
4. Across the bottom edge (direction `down` for each cell)
5. Back to the starting position

At corners, two boundary edges share the same cell but have different directions.

##### Hex Grid Boundary

The hex maze boundary has six edges corresponding to the six sides of the pointy-topped hexagon shape.

**Default entrance:** The cell at the vertical centre of the left edge (8 o'clock side), pointing `upper_left`. For a maze of height H, this is cell `(0, H/2)` with direction `upper_left`.

**Default exit:** Halfway around the boundary from the entrance. This places it on the right edge (2-4 o'clock side) at cell `(W-1, (W-1)/2 + (H-1)/2)` with direction `lower_right`.

**Traversal order:** Starting from the default entrance, trace clockwise through six edges:
1. Up the upper-left edge (10-12 o'clock): direction `upper_left` for each cell
2. Across the top edge (12-2 o'clock): direction `up` for each cell
3. Down the upper-right edge (2-4 o'clock): direction `lower_right` for each cell
4. Down the lower-right edge (4-6 o'clock): direction `lower_right` for each cell
5. Across the bottom edge (6-8 o'clock): direction `down` for each cell
6. Up the lower-left edge (8-10 o'clock): direction `upper_left` for each cell

Note: Edges 3-4 share the same direction (`lower_right`) but are on different sides of the hexagon. Similarly edges 1 and 6 share direction `upper_left`. The distinction is which cells they belong to.

*Details of exact cell sequences for each edge to be determined during implementation.*

#### Offset Calculation

The entrance and exit offsets are applied by stepping through the boundary sequence:

- Offset 0: Default position
- Positive offset N: Step N edges clockwise from default
- Negative offset N: Step |N| edges counter-clockwise from default

The offset wraps around the boundary if it exceeds the total number of boundary edges.

#### Interface

```mercury
    % boundary_edges(Topology, Bounds) = Edges
    %
    % Return the sequence of boundary edges tracing clockwise around the maze,
    % starting from the default entrance position.
    %
:- func boundary_edges(topology, bounds) = list(edge).

    % boundary_length(Topology, Bounds) = Length
    %
    % Return the number of edges in the boundary.
    % For a square maze: 2 * (W + H).
    % For a hex maze: 2 * W + 4 * H - 2 (to be verified).
    %
:- func boundary_length(topology, bounds) = int.

    % edge_at_offset(Topology, Bounds, Offset) = {Cell, Direction}
    %
    % Return the cell and outward direction at the given offset from the
    % default entrance position. Offset 0 is the default entrance.
    %
:- func edge_at_offset(topology, bounds, int) = {cell, direction}.
```

#### Usage in Rendering

The render module uses the boundary sequence to draw boundary walls:

1. Get the full boundary edge sequence
2. Find the indices of the entrance and exit edges
3. Convert edges to polyline points (corner coordinates)
4. Output two polylines: one from entrance to exit, one from exit to entrance (both going clockwise, leaving gaps at the openings)

### Option Handling

The options module handles command-line parsing using the standard library `getopt` module, and defines the `topology` type used by other modules.

#### Using getopt

The `getopt` module requires:

1. An enumeration type listing all options.
2. A predicate `short_option(char, option)` mapping single characters to options.
3. A predicate `long_option(string, option)` mapping long names to options.
4. A predicate `option_default(option, option_data)` specifying each option's type and default value.

Calling `process_options` with these predicates and the command-line arguments returns an `option_table` (a map from option to value) and the list of non-option arguments.

#### Types

**Topology:** The grid topology, defined here since it's a command-line option.

```mercury
:- type topology ---> square ; hex.
```

**Option:** Enumeration of all command-line options.

```mercury
:- type option
    --->    help
    ;       random_seed
    ;       size
    ;       dump.
    % ... more options as needed
```

**Options:** A record containing processed, validated options for convenient access.

```mercury
:- type options
    --->    options(
                topology :: topology,
                width :: int,
                height :: int,
                random_seed :: int,
                dump :: bool,
                output_file :: maybe(string)
            ).
```

#### Operations

- `parse_options(args, maybe_options, !io)` - Parse command-line arguments and return processed options, or an error.
- `usage(!io)` - Display usage information.

#### Design Notes

- The `options` record provides validated, typed values extracted from the raw `option_table`.
- Validation during parsing handles:
  - Converting size string (`N` or `WxH`) to width/height integers.
  - Parsing topology string to the `topology` type.
  - Ensuring output file is provided unless `--dump` is specified.
- The `options` record is passed to functions that need configuration values.

### Randomness

We use the `random.sfc16` generator from the standard library.

#### Rationale

- **Adequate quality** - For a maze game, we don't need cryptographic or statistical-simulation quality randomness. We just need mazes that look varied and unpredictable to human players.
- **Simplicity** - The ground style (`in`/`out` modes) is simple to work with. The RNG state is passed through generation predicates alongside the maze being built.
- **Efficiency** - Small state size means less overhead when passing it around.
- **No I/O coupling** - Maze generation is a pure computation and does not require I/O. Using the ground style avoids threading `!IO` through generation code.

#### Usage

- Derive the `sfc16` seed values from the command-line seed integer, or from the system clock if not provided.
- Pass the RNG state through maze generation predicates using regular `in`/`out` modes.
- Use `uniform_int_in_range` to pick random directions, cells, etc.

### Maze Generation

The `generate` module transforms an empty maze (no doors) into a complete maze with doors carved out.

#### Interface

```mercury
:- pred generate(maze::in, maze::out,
    random.sfc16.random::in, random.sfc16.random::out) is det.
```

- `maze::in` - The initial maze with no doors. Provides access to topology, bounds, start/target cells.
- `maze::out` - The completed maze with doors carved out.
- `random.sfc16.random` (in/out) - The RNG state, using `sfc16` directly (not via typeclass).

The `generate.dfs` submodule has the same signature:

```mercury
:- pred generate_dfs(maze::in, maze::out,
    random.sfc16.random::in, random.sfc16.random::out) is det.
```

#### Key Operations

- `unvisited_neighbours(maze, visited, cell) = list(cell)` - Get neighbours of a cell that are within the maze bounds and not yet visited.
- `choose_random(list(T), T, !RNG)` - Pick a random element from a non-empty list.

### Maze Checking

The `check` module validates that a maze is proper and returns the solution path.

#### Algorithm

1. Perform a depth-first traversal from the start cell:
   - Maintain `visited :: set_tree234(cell)` - the set of cells visited so far.
   - Maintain `solution :: maybe(list(cell))` - the solution path, if found.
   - Pass the current path in reverse order (leading back to the start cell).
   - For each cell visited:
     - If the cell is already in `visited`, throw an exception (loop detected).
     - Add the cell to `visited`.
     - If the cell is the target cell, reverse the current path to produce the solution.
     - Recurse to neighbours reachable through doors.

2. After the traversal completes:
   - If `count(visited) < count(cells(maze))`, throw an exception (unreachable cells).
   - If `solution` is `no`, throw an exception (no solution found).
   - Return the solution path.

#### Notes

- Exceptions are used for error reporting. Uncaught exceptions are sent to stderr, which suffices for development.
- In a proper maze, the DFS will visit every cell exactly once and find exactly one solution.
- The "no solution" case should never occur in a properly generated maze, but Mercury's determinism system requires handling it.

### Rendering

The `render` module generates an SVG depiction of a maze.

#### Parameters

- `S` - Cell spacing: the distance in viewbox units between the centres of adjacent cells (equivalently, the side length of a square cell).
- `M` - Margin: the margin around the maze, in units of cells.

#### Coordinate Mapping (Square Grids)

The viewbox origin is at the top-left of the image, with Y increasing downward (matching our grid convention).

**Cell corners:**

The top-left corner of cell `(0, 0)` is at viewbox coordinates `(M * S, M * S)`. For any cell `(X, Y)`:

- Top-left: `((M + X) * S, (M + Y) * S)`
- Top-right: `((M + X + 1) * S, (M + Y) * S)`
- Bottom-left: `((M + X) * S, (M + Y + 1) * S)`
- Bottom-right: `((M + X + 1) * S, (M + Y + 1) * S)`

**Edge endpoints:**

- A horizontal edge (up/down direction) between cells `(X, Y)` and `(X, Y+1)` runs from `((M + X) * S, (M + Y + 1) * S)` to `((M + X + 1) * S, (M + Y + 1) * S)`.
- A vertical edge (left/right direction) between cells `(X, Y)` and `(X+1, Y)` runs from `((M + X + 1) * S, (M + Y) * S)` to `((M + X + 1) * S, (M + Y + 1) * S)`.

**Boundary rectangle:**

For a maze of width `W` and height `H`:

- Top-left: `(M * S, M * S)`
- Bottom-right: `((M + W) * S, (M + H) * S)`
- Total viewbox size: `((2 * M + W) * S, (2 * M + H) * S)`

**Cell centre:**

For a cell `(X, Y)`, the centre is at `((M + X + 0.5) * S, (M + Y + 0.5) * S)`.

#### Coordinate Mapping (Hex Grids)

For hex grids, cell spacing `S` is the distance between adjacent cell centres. This equals the distance from the centre of a hexagon to each of its corners.

**Hex geometry constants:**

For a flat-topped regular hexagon with centre-to-corner distance `S`:
- Edge length: `S`
- Width (corner to corner): `2 * S`
- Height (edge to edge): `S * sqrt(3)`
- Horizontal distance between adjacent cell centres: `S * 1.5` (for `upper_right`/`lower_left` directions)
- Vertical distance between adjacent cell centres: `S * sqrt(3)` (for `up`/`down` directions)

**Cell centre:**

The centre of cell `(0, 0)` is placed at a position that accounts for the margin plus the offset needed to fit the pointy-topped maze shape. For a cell `(Q, R)`:

- The `upper_right` direction (increasing Q) moves by `(1.5 * S, -sqrt(3)/2 * S)`
- The `down` direction (increasing R) moves by `(0, sqrt(3) * S)`

Therefore, the centre of cell `(Q, R)` relative to cell `(0, 0)` is:
- X offset: `Q * 1.5 * S`
- Y offset: `R * sqrt(3) * S - Q * sqrt(3)/2 * S = (R - Q/2) * sqrt(3) * S`

*The exact formula including margin offset to be determined during implementation.*

**Cell corners:**

A flat-topped hexagon has 6 corners at angles 0°, 60°, 120°, 180°, 240°, 300° from the centre:
- Right corner (0°): `(centre_x + S, centre_y)`
- Upper-right corner (60°): `(centre_x + S/2, centre_y - S * sqrt(3)/2)`
- Upper-left corner (120°): `(centre_x - S/2, centre_y - S * sqrt(3)/2)`
- Left corner (180°): `(centre_x - S, centre_y)`
- Lower-left corner (240°): `(centre_x - S/2, centre_y + S * sqrt(3)/2)`
- Lower-right corner (300°): `(centre_x + S/2, centre_y + S * sqrt(3)/2)`

**Edge endpoints:**

Each hex edge connects two adjacent corners. The 6 edges correspond to the 6 directions:
- `up` edge: upper-left corner to upper-right corner
- `down` edge: lower-right corner to lower-left corner
- `upper_right` edge: upper-right corner to right corner
- `lower_left` edge: left corner to lower-left corner
- `upper_left` edge: left corner to upper-left corner
- `lower_right` edge: right corner to lower-right corner

**Viewbox dimensions:**

The viewbox is calculated as the bounding box of all cells plus margin padding.

1. Convert each of the six corner cells to screen coordinates (cell centres).
2. For each corner cell, account for the cell radius `S` to find the outermost points (the hex corners extend beyond the cell centre).
3. Find the minimum and maximum X and Y coordinates across all these points.
4. Add `M * S` padding on each side.

The viewbox dimensions are:
- Width: `(max_x - min_x) + 2 * M * S`
- Height: `(max_y - min_y) + 2 * M * S`

The origin is offset so that the top-left corner of the bounding box (before padding) is at `(M * S, M * S)`.

#### Implementation Notes

- SVG is generated by writing strings directly to a file (no XML library).
- Rounded line caps and joins are achieved using SVG attributes `stroke-linecap="round"` and `stroke-linejoin="round"`.
- Boundary walls are rendered as two polylines, each stopping at the entrance/exit openings.
- Coordinates may be floating-point (e.g., if S is odd, cell centres will have fractional coordinates). SVG handles this without issue.

#### Interface

```mercury
:- pred render_maze(string::in, maze::in, list(cell)::in, io::di, io::uo) is det.
```

- `string` - The output filename.
- `maze` - The maze to render.
- `list(cell)` - The solution path (may be empty to omit the solution).

Default render parameters are used initially; a `render_options` type can be added later.

## Algorithm Details

### DFS Generator

The DFS (Depth-First Search) generator, also known as Recursive Backtracker, carves a maze by performing a randomised depth-first traversal.

#### Algorithm

1. Start from the start cell, mark it as visited.
2. While there are cells on the stack:
   a. If the current cell has unvisited neighbours:
      - Randomly choose one.
      - Carve a door between current and chosen cell.
      - Push current cell onto the stack.
      - Move to the chosen cell and mark it as visited.
   b. Otherwise (dead end):
      - Pop a cell from the stack and make it current (backtrack).
3. When the stack is empty, all reachable cells have been visited.

#### State

- `visited :: set_tree234(cell)` - Cells that have been visited.
- `stack :: list(cell)` - Backtrack stack (the current path from start).
- `doors :: set_tree234(edge)` - Doors carved so far.
- `rng :: sfc16` - Random number generator state.

#### Characteristics

Produces mazes with long, winding corridors and relatively few dead ends. The algorithm creates a bias toward longer paths.

### Kruskal's Generator

Kruskal's algorithm builds a maze by randomly removing walls, using a union-find data structure to ensure no loops are created.

#### Algorithm

1. Create a list of all potential internal walls (edges between adjacent cells).
2. Shuffle the list randomly.
3. For each wall in the list: if the cells on either side belong to different sets (not yet connected), remove the wall (add a door) and merge their sets.

#### Union-Find Data Structure

The union-find (disjoint set) data structure efficiently tracks which cells are connected. It supports two operations:

- **Find(cell)** - returns the "representative" of the set containing this cell.
- **Union(cell1, cell2)** - merges the sets containing these two cells.

A simple implementation uses a map from each cell to its parent. Initially each cell is its own parent (its own set).

- **Find** follows parent pointers until reaching a cell that is its own parent (the representative).
- **Union** finds the representatives of both cells and makes one point to the other.

#### Implementation

```mercury
:- type union_find == map(cell, cell).

:- func init_union_find(list(cell)) = union_find.
:- func find(union_find, cell) = cell.
:- pred union(cell, cell, union_find, union_find).
```

For small mazes, a simple implementation without path compression or union-by-rank is sufficient.

#### Characteristics

Produces more uniform mazes with less bias than DFS. The maze grows from many points simultaneously rather than from a single path.

### Prim's Generator

Prim's algorithm grows a maze outward from a starting cell by repeatedly adding frontier cells.

#### Algorithm

1. Start with the start cell in the maze; add all its neighbours to the frontier set.
2. While the frontier is not empty:
   a. Pick a random cell from the frontier.
   b. Find all its neighbours that are already in the maze.
   c. Pick one randomly and carve a door between them.
   d. Add the frontier cell to the maze.
   e. Add all unvisited neighbours of the new cell to the frontier.

#### Data Structures

- **In-maze set** (`set_tree234(cell)`) - tracks which cells are already part of the maze.
- **Frontier set** (`set_tree234(cell)`) - unvisited cells adjacent to cells in the maze.

The frontier is a set rather than a list to avoid duplicates (a cell might be adjacent to multiple in-maze cells).

#### Implementation

```mercury
:- pred generate_prim(maze::in, maze::out,
    random.sfc16.random::in, random.sfc16.random::out) is det.
```

Key operations:
- Pick a random element from the frontier set (convert to list, choose randomly).
- Find in-maze neighbours of the chosen cell.
- Carve a door to one of them.
- Update frontier by adding unvisited neighbours.

#### Characteristics

Produces mazes similar to Kruskal's but with a different growth pattern—the maze expands outward from a central region. Tends to have shorter dead ends and more branching near the starting point.

## Implementation Strategy

We will develop the program in iterations, filling in detailed requirements and design as we go.

### Iteration 1 (Complete)

**Goals:**

1. Implement the following modules:

   - `grid`
   - `maze`
   - `options`
   - `dump`
   - `amaze`

2. The program should:

   - Parse the options we have defined, using fixed dummy values for options not yet implemented.
   - Construct an empty square maze with no doors, using the size from options (default 10x10).
   - Dump the maze to standard output (in later iterations we will only do this if `--dump` is given).

### Iteration 2 (Complete)

**Goals:**

1. Implement the following modules:

   - `check`
   - `generate`
   - `generate.dfs`

2. Update the maze module:

   - Instead of storing entrance and exit edges, store the start cell, target cell, and their directions towards the outside of the maze.

3. The program should:

   - Process command-line options.
   - Seed the random number generator from the command-line argument, or from system time if not given.
   - Determine the start cell and target cell from the bounds.
   - Generate the set of doors for the maze using the DFS algorithm (hard-coded in `generate`).
   - Check the maze (validate and find the solution path).
   - If there are errors, print them to stderr; otherwise dump the maze and solution to stdout.

### Iteration 3 (Complete)

**Goals:**

1. Implement the `render` module.

2. If the `--dump` option is given, the program should behave as in iteration 2.

3. If the `--dump` option is not given, the program should:

   - Generate and check the maze as in iteration 2, throwing an exception on errors as we currently do.
   - Generate a depiction of the maze in `output.svg`, using default values for the output parameters.

### Iteration 4 (Complete)

**Goals:**

1. Define all options and defaults.

2. Fully implement the options module with the additional options.

3. Determine correct entrance and exit locations.

4. Use the output parameters when rendering the SVG.

### Iteration 5 (Complete)

**Goals:**

1. Implement the remaining maze generation algorithms.

**Algorithms:**

- [x] Recursive Backtracker (DFS)
- [x] Kruskal's Algorithm
- [x] Prim's Algorithm
- [x] Aldous-Broder Algorithm
- [x] Wilson's Algorithm
- [x] Hunt-and-Kill Algorithm
- [x] Binary Tree Algorithm
- [x] Sidewinder Algorithm

### Iteration 6 (Complete)

**Goals:**

1. Support hexagonal grid topology.

**Design Tasks:**

- [x] Define hex cell coordinate system and geometry
- [x] Define hex bounds interpretation
- [x] Define hex boundary traversal (clockwise order)
- [x] Define default entrance/exit positions for hex grids
- [x] Define viewbox dimensions for hex grids
- [x] Review generation algorithms for hex compatibility

**Implementation Tasks:**

- [x] Update grid.m: hex adjacency with basis vectors (upper_right, down)
- [x] Update maze.m: implement `in_maze` for hex bounds
- [x] Update maze.m: implement `cells` for hex (enumerate all cells in bounds)
- [x] Update maze.m: implement `internal_edges` for hex
- [x] Implement hex_boundary_walls in boundary.m (one entry per edge, not per cell)
- [x] Implement hex coordinate geometry helpers in render.m
- [x] Update render_cell_background for hex (draw boundary polygon)
- [x] Update edge_endpoints for hex
- [x] Update wall_start_point / wall_end_point for hex
- [x] Update cell_centre for hex
- [x] Update viewbox calculation for hex
- [x] Test with topology-agnostic algorithms (DFS, Kruskal, Prim, Aldous-Broder, Wilson, Hunt-and-Kill)
- [x] Add error handling for Binary Tree and Sidewinder with hex topology
- [x] Add direction_delta to grid.m for consistent basis handling
- [x] Update size option interpretation for hex (N means W=2N-1, H=N)

### Iteration 7 (Complete)

**Goals:**

Add final polish to the project.

**Tasks:**

- [x] Review codebase for clarity and modularity
- [x] Implement refactorings and cleanup
- [x] Graceful error handling
- [x] Write README.md
